/////////////////////////////////////////////////////////////////////////////
//
// Format: VB2
//
// Used by: DDR, Castlevania Chronicles, other Konami games
//
/////////////////////////////////////////////////////////////////////////////

#include "headers.h"

#include "fmt.h"
#include "rip.h"

/////////////////////////////////////////////////////////////////////////////

int fmt_vb2_detect(FILE *inf, const char *ext) {
  int score = 50;

  // if it ends in vb2, we're pretty sure
  if(!strcasecmp(ext, "vb2")) return 90;

  // otherwise, it's not so clear - would have to search around for the
  // start of the data

  return score;
}

/////////////////////////////////////////////////////////////////////////////
//
// Tell if a block of bytes is all zero
//
int iszeroes(const uint8 *buf, uint32 n) {
  uint32 i;
  for(i = 0; i < n; i++) {
    if(buf[i]) return 0;
  }
  return 1;
}

/////////////////////////////////////////////////////////////////////////////
//
// Detect the beginning of a vb2 music block
// Returns the interleave size if successful
// Returns 0 if no beginning was detected
//
static uint32 detect_vb2_beginning(
  const uint8 *buf,
  uint32 size
) {
  if(!buf) return 0;
  //
  // DDR, various versions
  //
  if(size >= 0x8000) {
    if(
      (buf[0x0001] == 0x00) &&
      (buf[0x0011] == 0x06) &&
      (buf[0x4001] == 0x00) &&
      (buf[0x4011] == 0x06)
    ) return 0x4000;
  }
  //
  // DDRMAX2
  //
  if(size >= 0x4000) {
    if(
      iszeroes(buf+0x0000, 0x10) &&
      ((buf[0x0011] == 0x06) || (buf[0x0011] == 0x04)) &&
      iszeroes(buf+0x2000, 0x10) &&
      ((buf[0x2011] == 0x06) || (buf[0x2011] == 0x04))
    ) return 0x2000;
  }
  //
  // Castlevania Chronicles (J)
  //
  if(size >= 0x8000) {
    if(
      ((buf[0x0000] == 0x00) && ( iszeroes(buf + 0x0002, 0x0E))) &&
      ((buf[0x4000] == 0x00) && ( iszeroes(buf + 0x4002, 0x0E))) &&
      ((buf[0x0010] != 0x00) || (!iszeroes(buf + 0x0012, 0x0E))) &&
      ((buf[0x4010] != 0x00) || (!iszeroes(buf + 0x4012, 0x0E))) &&
      ((buf[0x0011] == 0x02) || (buf[0x0011] == 0x06)) &&
      ((buf[0x4011] == 0x02) || (buf[0x4011] == 0x06)) &&
       (buf[0x0001] == buf[0x4001]) &&
       (buf[0x0011] == buf[0x4011])
    ) return 0x4000;
  }
  //
  // Castlevania Chronicles (US)
  //
  if(size >= 0x10000) {
    if(
      ((buf[0x0000] == 0x00) && ( iszeroes(buf + 0x0002, 0x0E))) &&
      ((buf[0x8000] == 0x00) && ( iszeroes(buf + 0x8002, 0x0E))) &&
      ((buf[0x0010] != 0x00) || (!iszeroes(buf + 0x0012, 0x0E))) &&
      ((buf[0x8010] != 0x00) || (!iszeroes(buf + 0x8012, 0x0E))) &&
      ((buf[0x0011] == 0x02) || (buf[0x0011] == 0x06)) &&
      ((buf[0x8011] == 0x02) || (buf[0x8011] == 0x06)) &&
       (buf[0x0001] == buf[0x8001]) &&
       (buf[0x0011] == buf[0x8011])
    ) return 0x8000;
  }

  // all tests failed
  return 0;
}

/////////////////////////////////////////////////////////////////////////////

static int blockdetect(long ofs, const uint8 *block, uint32 blocksize) {
  if(!block) return 0;
  if(detect_vb2_beginning(block, blocksize)) return 1;
  return 0;
}

/////////////////////////////////////////////////////////////////////////////

int fmt_vb2_rip(
  FILE *inf,
  struct OPTIONS *opts
) {
  int errors = 0;
  uint32 bytesskipped = 0;
  uint8 buf[0x10000];
  struct OPTIONS myopts;
  long startpos;
  long addpos;
  long searchstartpos = -1;
  long lastsongendedat = 0;
  int songnum = 0;
  const char *orig_outname;
  char *myoutname = NULL;

  if(opts) { memcpy(&myopts, opts, sizeof(myopts)); }
  else { memset(&myopts, 0, sizeof(myopts)); }

  orig_outname = myopts.outname;
  if(!orig_outname) orig_outname = "out";

  myoutname = malloc(strlen(orig_outname) + 20);
  if(!myoutname) {
    printf("unable to allocate string\n");
    goto error;
  }

  startpos = ftell(inf);

  if(!myopts.use_channels) {
    myopts.use_channels = 1;
    myopts.channels = 2;
  }
  if(!myopts.use_honor_endflag) {
    myopts.use_honor_endflag = 1;
    myopts.honor_endflag = 1;
  }

  addpos = 0;
  for(;;) {
    int r;
    uint32 interleave;
    fseek(inf, startpos + addpos, SEEK_SET);
    r = fread(buf, 1, 0x10000, inf);
    fseek(inf, startpos + addpos, SEEK_SET);
    if(r < 0) {
      printf("%s\n", strerror(errno));
      goto error;
    }
    if(!r) break;
    interleave = detect_vb2_beginning(buf, r);
    if(!interleave) {
      if(addpos >= lastsongendedat) {
        if(searchstartpos < 0) {
          searchstartpos = ftell(inf);
          printf("searching at 0x%lX...", searchstartpos);
          fflush(stdout);
        }
        bytesskipped += 0x800;
      }
      addpos += 0x800;
      continue;
    }
    if(searchstartpos >= 0) {
      printf("skipped %lu bytes\n", ftell(inf) - searchstartpos);
      searchstartpos = -1;
    }
    myopts.use_interleave = 1;
    myopts.interleave = interleave;

    sprintf(myoutname, "%s%04d", orig_outname, songnum);
    myopts.outname = myoutname;
    songnum++;

    errors += rip(inf, &myopts, blockdetect);

    //
    // rather than advancing completely beyond this song, we should seek back
    // 2*interleave to make sure the last block isn't short/malformed.
    // this fixes DDRMAX/DDRMAX2
    //
    lastsongendedat = (ftell(inf) - startpos) & (~0x7FF);
    { long newaddpos = lastsongendedat - 2 * interleave;
      if(newaddpos <= addpos) { newaddpos = addpos + 0x800; }
      addpos = newaddpos;
    }
  }

  if(searchstartpos >= 0) {
    printf("nothing found\n");
  }

  printf("total bytes skipped: %u\n", bytesskipped);

  goto end;
error:
  errors++;
end:
  if(myoutname) free(myoutname);
  return errors;
}

/////////////////////////////////////////////////////////////////////////////
