# GNU Enterprise Forms - wx 2.6 UI Driver - Page widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: page.py 9956 2009-10-11 18:54:57Z reinhard $
"""
Implementation of a <page> tag in the UI layer.
"""

import wx

from gnue.forms.uidrivers.wx26.widgets import _base

__all__ = ['UIPage']

# =============================================================================
# Interface implementation of a page
# =============================================================================

class UIPage(_base.UIHelper):
    """
    UI widget for the <page> tag.

    @ivar page_index: zero-based index of the page within the form's
        page-container (which is either a notebook or a panel).
    """

    page_index = 0

    # -------------------------------------------------------------------------
    # Create a new page widget
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):
        """
        Create the wx.Panel for the page and add it to the page-container of
        the form.  The spacer is ignored for page tags.

        @param event: the creation-event instance carrying information like
            container (parent-widget)
        @param spacer: not used for pages

        @returns: the wx.Panel instance which is the parent for all child
            widgets of this page.  This instance is already added to the
            page-container of the form.
        """

        self.page_index = len(self._uiForm.pages)

        parent = event.container

        self._container = wx.Panel(parent, -1)

        if self.managed:
            page_sizer = wx.BoxSizer(wx.VERTICAL)
        else:
            page_sizer = wx.GridBagSizer()
            page_sizer.SetEmptyCellSize((self._uiDriver.cellWidth,
                    self._uiDriver.cellHeight))
            w = getattr(self._form._layout, 'Char__width', 0)
            h = getattr(self._form._layout, 'Char__height', 0)
            width = w and (self._uiDriver.cellWidth * w) or -1
            height = h and (self._uiDriver.cellHeight * h) or -1

            page_sizer.SetMinSize((width, height))

        self._container.SetSizer(page_sizer)

        if isinstance(parent, wx.Notebook):
            title = "%s" % (self._gfObject.caption or self._gfObject.name)
            parent.AddPage(self._container, title, False)
        else:
            parent.GetSizer().Add(self._container, 1, wx.EXPAND)

        self._uiForm.pages.append(self._container)

        return self._container


    # -------------------------------------------------------------------------
    # Add widgets to the page
    # -------------------------------------------------------------------------

    def add_widgets(self, ui_widget, spacer, border=0):
        """
        Add a given UI widget to the page.

        @param ui_widget: widget to add to the page
        @param spacer: not used for pages
        """

        item = ui_widget.widget
        sizer = self._container.GetSizer()

        if self.managed:
            # The border between the edge of the page and the nearest control
            # is 12 pixel according to GNOME Human Interface Guidlines.
            sizer.Add(item, ui_widget.stretch, wx.EXPAND | wx.ALL, 12)
        else:
            flags = wx.ALIGN_CENTER | wx.EXPAND
            if not ui_widget.is_growable():
                box = wx.BoxSizer(wx.HORIZONTAL)
                box.Add(item, 1, wx.ALIGN_CENTER_VERTICAL)
                item = box

            if border:
                flags |= wx.TOP | wx.BOTTOM

            sizer.Add(item, ui_widget.chr_pos, ui_widget.chr_span, flags,
                    border)



# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIPage,
  'provides' : 'GFPage',
  'container': 1
}
