# GNU Enterprise Navigator - QT Frontent
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: UIqt3.py 8129 2006-01-18 21:25:44Z jcater $

import qt

import urllib
import urlparse
import os

from gnue.common.apps import GConfig
from gnue.common.utils.FileUtils import openResource
from gnue.navigator import VERSION
from gnue.navigator import GNParser

try:
  from gnue.forms.GFInstance import GFInstance
  from gnue.forms.uidrivers import qt3 as ui

except ImportError:
  FORMS_SUPPORT = False
  print 'GNUe Forms is not installed on your system'

images_dir = GConfig.getInstalledBase ('forms_images','common_images') + '/'


# =============================================================================
# This class implements the GTK-UI of the navigator client
# =============================================================================

class Instance:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, processes):
    """
    @param processes: GNObjects tree describing the current loaded gpd
    """

    self.processes         = processes
    self._lastSerialNumber = 0

    self.__lastItem        = ""

    self.app = ui.getQtApp ()

         
  # ---------------------------------------------------------------------------
  # Build the UI and start the navigator client
  # ---------------------------------------------------------------------------

  def run (self, instance):
    """
    This function creates the user interface and starts the main loop
    """

    self.__instance = instance

    self.__buildInterface ()
    
    self.__buildTreeModel(self.processes)

    self.processes.setClientHandlers ({'form': self.runForm})

    self.mainWindow.show ()
    self.app.exec_loop ()
  

  # ---------------------------------------------------------------------------
  # display a message in the status bar
  # ---------------------------------------------------------------------------

  def setStatus (self, message = ""):
    """
    This function removes the last message from the status bar and adds a new
    one if specified.

    @param message: message to put into the status bar or None
    """

    self.__statusbar.message(message)


  # ---------------------------------------------------------------------------
  # Build up the user interface
  # ---------------------------------------------------------------------------

  def __buildInterface (self):
    """
    This function creates the user interface and connects all signal handlers
    """
   
    self.mainWindow = qt.QMainWindow()
    self.mainWidget = qt.QWidget(self.mainWindow)
    self.mainWindow.setCentralWidget(self.mainWidget)
    base_sizer = qt.QVBoxLayout(self.mainWidget)
    base_sizer.setMargin(6)
    base_sizer.setResizeMode(qt.QLayout.Minimum)
    self.mainWindow.setCaption ('GNUe Navigator')

    # StatusBar
    self.__statusbar = self.mainWindow.statusBar()
    
    # Menu-Bar    
    self.__menubar = self.mainWindow.menuBar()    
    self.__createMenuBar ()
    
    ctsplitter = qt.QSplitter(self.mainWidget,"Content Splitter")
    ctsplitter.setOrientation(qt.QSplitter.Horizontal)

    self.__treeView = qt.QListView(ctsplitter,"Tree View")
    self.__treeView.addColumn(u_("Process"))
    qt.QObject.connect(self.__treeView, \
                       qt.SIGNAL('clicked (QListViewItem *)'), \
                       self.__tree_item_selected)
    qt.QObject.connect(self.__treeView, \
                       qt.SIGNAL('spacePressed (QListViewItem *)'), \
                       self.__tree_item_selected)
    qt.QObject.connect(self.__treeView, \
                       qt.SIGNAL('doubleClicked (QListViewItem *)'), \
                       self.__tree_item_activated)
    qt.QObject.connect(self.__treeView, \
                       qt.SIGNAL('returnPressed (QListViewItem *)'), \
                       self.__tree_item_activated)    

    self.__display = qt.QTextEdit(ctsplitter,"Content Display")
    self.__display.setText(self.__getTitlePage ())
    base_sizer.addWidget(ctsplitter)

    self.mainWindow.resize(qt.QSize(600,400).expandedTo( \
                        self.mainWindow.minimumSizeHint()))
    self.mainWindow.clearWState(qt.Qt.WState_Polished)
    

  # ---------------------------------------------------------------------------
  # Create the menu bar
  # ---------------------------------------------------------------------------

  def __createMenuBar (self):
    """
    This function creates the menu bar for the navigator.
    """
    
    menu = qt.QPopupMenu(self.__menubar)
    self.__menubar.insertItem(u_("&File"), menu)
    self._tasks={}
    self._tasks[menu.insertItem(u_('&Open'))]=self.__openFile
    self._tasks[menu.insertItem(u_('&Quit'))]=self.__windowExit
    qt.QObject.connect(menu, qt.SIGNAL('activated(int)'),
                self.__menuitemSelected)


    menu = qt.QPopupMenu(self.__menubar)
    self.__menubar.insertItem(u_("&Favorites"), menu)
    self._tasks[menu.insertItem(u_('&Add Favorite'))] = lambda :0
    self._tasks[menu.insertItem(u_('&Organize Favorites'))] = lambda :0
    qt.QObject.connect(menu, qt.SIGNAL('activated(int)'),
                self.__menuitemSelected)

    menu = qt.QPopupMenu(self.__menubar)
    self.__menubar.insertItem(u_("&Help"), menu)
    item = menu.insertItem(u_('&About'))
    qt.QObject.connect(menu, qt.SIGNAL('activated(int)'),
                self.__about)
    
    #menu.setAccel(qt.QKeySequence(hotkey), item)
    #menu.setWhatsThis(mid, userAction.description or '')
    
  def __menuitemSelected(self,id):
    if self._tasks.has_key(id):
      self._tasks[id]()
    
  
  # ---------------------------------------------------------------------------
  # Close the navigator
  # ---------------------------------------------------------------------------

  def __windowExit (self):
    """
    This function quits the main loop
    """

    # TODO: can we keep track of all windows opened ?
    self.app.quit ()


  # ---------------------------------------------------------------------------
  # Create a new tree model
  # ---------------------------------------------------------------------------

  def __buildTreeModel (self, GNTree):
    """
    This function creates a new tree store and populates the given GNObjects
    tree into that store

    @param GNTree: GNObjects tree to be added to the QListView
    """

    self.__treeView.clear()
    GNTree.walk (self.__addToTreeModel)


  # ---------------------------------------------------------------------------
  # Add an item of a GNObjects tree to the TreeStore
  # ---------------------------------------------------------------------------

  def __addToTreeModel (self, gnObject):
    """
    This function adds an item of a GNObjects tree to the given tree 

    @param gnObject: GNObject instance to be added
    """
        
    if gnObject._type == 'GNProcesses':
      node = qt.QListViewItem(self.__treeView, gnObject.title)
      node.setOpen(1)
      

    elif gnObject._type in ['GNStep', 'GNProcess']:
      node = qt.QListViewItem(gnObject.getParent ().__node, gnObject.title)      
      node.setOpen(0)
      node._gnobject = gnObject

    else:
      return

    # Remember the iterator of the current node
    gnObject.__node = node


  # ---------------------------------------------------------------------------
  # Activate a row in the tree view
  # ---------------------------------------------------------------------------

  def __tree_item_activated (self, item):
    """
    This function is called on the 'row-activated' signal of the tree view,
    which happens if an element of the tree get's selected by the enter-key or
    a double-click.

    @param tree: the tree view widget which emitted the signal
    @param path: the path tuple of the selected item
    @param column: the tree view column instance which has been activated
    """

    try:
      self.beginWait ()

      item.setOpen(item.isOpen() and 0 or 1)

      # Fetch the associated GN* instance, which is hold in the first column
      if hasattr(item,"_gnobject"):
        gnObject = item._gnobject
        if gnObject._type == 'GNStep':
          gnObject.run ()

    finally:
      self.endWait ()


  # ---------------------------------------------------------------------------
  # A row has been selected in the tree
  # ---------------------------------------------------------------------------

  def __tree_item_selected (self, item):
    """
    This function get's called when the focus in the tree view widget has
    changed.

    @param tree: the tree view widget which emitted the signal
    """

    try:
      self.beginWait ()
    
      if item != self.__lastItem:
        self.__lastItem = item
        
        if hasattr(item,"_gnobject"):
          gnObject = item._gnobject
          descr    = gnObject.findChildOfType ('GNDescription')
  
          if descr is not None:
            stream = descr.getChildrenAsContent ()
            stream = '''<html><head><meta http-equiv="Content-Type" content="text/html; charset=utf8">
                      </head><body>%s</body></html>''' % stream
          else:
            stream = self.__getTitlePage ()
  
          self.__display.setText(stream)

    finally:
      self.endWait ()


  # ---------------------------------------------------------------------------
  # Create a HTML stream with the title page
  # ---------------------------------------------------------------------------

  def __getTitlePage (self):
    """
    This function creates a HTML string containing the title page

    @return: stream with the HTML code for the titlepage
    """

    return '<HTML><BODY><CENTER><B>GNUe Navigator</B>' \
           '<p><img src="%s"></p>' \
           '<p>A part of the <a href="http://www.gnuenterprise.org/">' \
           'GNU Enterprise Project</a></p>' \
           '</center></body></html>' % (images_dir + "/ship2.png")


  # --------------------------------------------------------------------------
  # Change the mouse pointer in an hour-glass
  # --------------------------------------------------------------------------

  def beginWait (self):
    """
    This function changes the mouse pointer to an hour glass.
    """

    self.mainWindow.setCursor(qt.QCursor(qt.Qt.WaitCursor))


  # --------------------------------------------------------------------------
  # Change the mouse pointer back to it's normal apperance
  # --------------------------------------------------------------------------

  def endWait (self):
    """
    This function changes the mouse pointer back to the normal state.
    """

    self.mainWindow.setCursor(qt.QCursor(qt.Qt.ArrowCursor))    

  # ---------------------------------------------------------------------------
  # Run a form
  # ---------------------------------------------------------------------------

  def runForm (self, step, parameters = {}):
    """
    This function runs a GNStep with a GNUe Form definition.

    @param step: the GNStep instance describing the form
    @param paramters: dictionary with user parameters for the form
    """

    # TODO: this code *should* be executed in a new thread.
    if os.path.basename (step.location) == step.location:
      try:
        formdir = gConfigNav ('FormDir')

      except KeyError:
        formdir = ""

      formfile = os.path.join (formdir, step.location)

    else:
      formfile = step.location

    try:
      self.beginWait ()
      self.setStatus ('running form %s' % formfile)
      self._runForm (formfile, parameters)

    finally:
      self.endWait ()
      self.setStatus ()


  # ---------------------------------------------------------------------------
  # Run a form from a trigger
  # ---------------------------------------------------------------------------

  def runFormFromTrigger (self, form, parameters = {}):
    """
    This function runs a form from trigger code.

    @param form: URL of the form
    @param parameters: dictionary with user parameters for the form
    """

    try:
      self.beginWait ()
      self.setStatus ('running form %s from trigger' % formfile)

      self._runForm (form, self._params)

    finally:
      self.endWait ()
      self.setStatus ()


  # ---------------------------------------------------------------------------
  # Do the dirty work of running a form
  # ---------------------------------------------------------------------------

  def _runForm (self, formfile, parameters):
    """
    This function starts a new form.

    @param formfile: URL of the form to be started
    @param parameters: dictionary with user parameters for the form
    """

    self.__instance.run_from_file(formfile, parameters)


  # ---------------------------------------------------------------------------
  # Open another process definition file
  # ---------------------------------------------------------------------------

  def __openFile (self):
    """
    This function displays a 'FileSelection' dialog to select other GPD files.
    If a file was selected it will be loaded into a new process tree.
    """

    filename = str(qt.QFileDialog.getOpenFileName("/home",
                    u_("GNUe Process Definition (*.gpd)"), self.mainWindow,
                    u_("Open Process Definition"),
                    u_("Select another process definition file")))
  
    self.__reload(filename)

  # ---------------------------------------------------------------------------
  # Load another process definition file
  # ---------------------------------------------------------------------------

  def __reload (self, filename):
    """
    This function loads a new process tree from the given filename and replaces
    the current tree view with the new process tree.

    @param filename: name of the GPD file to be loaded
    """

    fhd = openResource (filename)

    try:
      processes = GNParser.loadProcesses (fhd)
      processes._connections   = self.processes._connections
      processes._configManager = self.processes._configManager
      processes._ui_type       = self.processes._ui_type
       
      self.processes = processes
      
    finally:
      fhd.close ()


  # ---------------------------------------------------------------------------
  # Display a nice about message
  # ---------------------------------------------------------------------------

  def __about (self):
    """
    This function displays an about box.
    """

    text = u_("GNUE Navigator")+"\n"+      \
    _("    Version : ")+"%s\n"+         \
    _("    Driver  : UIgtk2")+"\n"+ \
    _("-= Process Info =-                  ")+"\n"+        \
    _("    Name   : ")+"%s\n"+          \
    _("    Version: ")+"%s\n"+          \
    _("    Author : ")+"%s\n"+          \
    _("    Description: ")+"%s\n"
    
    qt.QMessageBox.information(self.mainWindow, u_("About"), 
                               text % (VERSION,"","","",""))
