/*
 * arch/arm/mach-dmw/timer.c
 *
 * Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/clockchips.h>
#include <linux/clocksource.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/gpio.h>
#include <linux/profile.h>

#include <asm/mach/time.h>
#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/irqs.h>
#include <mach/clock.h>
#include <mach/timing.h>
#include <mach/reset.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#define DMW_RTC_CFG		0x00
#define DMW_RTC_CTL		0x04
#define DMW_RTC_ALARMVAL	0x08
#define DMW_RTC_STATUS		0x0C
#define DMW_RTC_INTCLR		0x10
#define DMW_RTC_TIME		0x14

#define CFG_ALARMEN		(1 << 0)
#define CFG_INTEN		(1 << 1)
#define CTL_SWRST		(1 << 0)
#define STATUS_WRITEALARMVALEN	(1 << 1)
#define INTCLR_ALARM		(1 << 0)

#define DMW_TMR_LOAD	0x00
#define DMW_TMR_VALUE	0x04
#define DMW_TMR_CTRL	0x08
#define DMW_TMR_INTCLR	0x0c

#define CTRL_ONESHOT	(1 << 0)
#define CTRL_32BIT	(1 << 1)
#define CTRL_DIV1	(0 << 2)
#define CTRL_DIV16	(1 << 2)
#define CTRL_DIV256	(2 << 2)
#define CTRL_IE		(1 << 5)
#define CTRL_PERIODIC	(1 << 6)
#define CTRL_ENABLE	(1 << 7)

static void *rtc_base;

static unsigned long rtc_read(void)
{
	unsigned long val1, val2;

	val2 = readl(rtc_base + DMW_RTC_TIME);
	do {
		val1 = val2;
		val2 = readl(rtc_base + DMW_RTC_TIME);
	} while (val1 != val2);

	return val2;
}

static cycle_t
dmw_get_cycles(struct clocksource *cs)
{
	return rtc_read();
}

static cycle_t
dmw_get_cycles_dummy(struct clocksource *cs)
{
	return 0;
}

static struct clocksource rtc_clocksource = {
	.name		= "rtc",
	.rating		= 100,
	.read		= dmw_get_cycles_dummy,
	.mask		= CLOCKSOURCE_MASK(32),
	.shift		= 17,
	.flags		= CLOCK_SOURCE_IS_CONTINUOUS | CLOCK_SOURCE_VALID_FOR_HRES,
};

static void
rtc_set_mode(enum clock_event_mode mode, struct clock_event_device *clk)
{
	/* nothing to do */
}

static int
rtc_set_next_event(unsigned long evt, struct clock_event_device *clk)
{
	unsigned long flags;

	if (evt < 1)
		evt = 2;

	/* Wait until we can set the alarm */
	while (!(readl(rtc_base + DMW_RTC_STATUS) & STATUS_WRITEALARMVALEN))
		; /* busy wait */

	local_irq_save(flags);
	local_fiq_disable();
	writel(rtc_read() + evt, rtc_base + DMW_RTC_ALARMVAL);
	local_irq_restore(flags);

	return 0;
}

static struct clock_event_device rtc_clockevent = {
	.name		= "rtc",
	.shift		= 17,
	.features       = CLOCK_EVT_FEAT_ONESHOT,
	.set_mode	= rtc_set_mode,
	.set_next_event	= rtc_set_next_event,
};

static irqreturn_t
dmw_rtc_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *evt = dev_id;

	writel(INTCLR_ALARM, rtc_base + DMW_RTC_INTCLR);

	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static struct irqaction dmw_rtc_irq = {
	.name		= "rtc",
	.flags		= IRQF_TIMER,
	.handler	= dmw_rtc_interrupt,
	.dev_id		= &rtc_clockevent,
};

/*
 * Returns current time from boot in nsecs. It's OK for this to wrap
 * around for now, as it's just a relative time stamp.
 */
unsigned long long sched_clock(void)
{
	return clocksource_cyc2ns(rtc_clocksource.read(&rtc_clocksource),
				  rtc_clocksource.mult, rtc_clocksource.shift);
}

/**
 * read_persistent_clock -  Return time from a persistent clock.
 *
 * Reads the time from a source which isn't disabled during PM, the
 * 32k sync timer.  Convert the cycles elapsed since last read into
 * nsecs and adds to a monotonically increasing timespec.
 */
void read_persistent_clock(struct timespec *ts)
{
	static struct timespec persistent_ts;
	static cycles_t cycles, last_cycles;

	unsigned long long nsecs;
	cycles_t delta;
	struct timespec *tsp = &persistent_ts;

	last_cycles = cycles;
	cycles = rtc_clocksource.read(&rtc_clocksource);
	delta = cycles - last_cycles;

	nsecs = clocksource_cyc2ns(delta, rtc_clocksource.mult, rtc_clocksource.shift);
	timespec_add_ns(tsp, nsecs);
	*ts = *tsp;
}


#ifdef CONFIG_DMW_IRQMEAS
static unsigned long irq_timer_freq;
static void *irq_timer_base;
struct timing irq_timings[DMW_PLICU_NR_IRQS];
static unsigned char *irq_strings[DMW_PLICU_NR_IRQS]={DMW_IRQ_STRINGS};


#define IRQ_NESTING_STACK_SIZE 1024

typedef struct  stack{
	unsigned int tos;
	unsigned int stack[IRQ_NESTING_STACK_SIZE];
}irq_stack_t;

static irq_stack_t irqtiming_stack[2];

unsigned int pushirq(unsigned int val, unsigned int i)
{
	if(irqtiming_stack[i].tos == IRQ_NESTING_STACK_SIZE)
	{
		return 0;
	}
	else
	{
		irqtiming_stack[i].stack[irqtiming_stack[i].tos++] = val;
		return 1;
	}
}

unsigned int popirq( unsigned int *pval, unsigned int i)
{
	if(irqtiming_stack[i].tos == 0)
	{
		return 0;
	}
	else
	{
		*pval = irqtiming_stack[i].stack[--irqtiming_stack[i].tos];
		return 1;
	}
}
unsigned int is_irqstack_empty(unsigned int i)
{
	return (irqtiming_stack[i].tos == 0);
}

static void init_irq_stack(unsigned int i)
{
	irqtiming_stack[i].tos = 0;
}



unsigned long timing_get(void)
{
	return readl(irq_timer_base + DMW_TMR_VALUE);
}

unsigned long timing_calc(unsigned long cnt0, unsigned long cnt1)
{
	return cnt0 > cnt1 ? cnt0 - cnt1 : cnt0 + ~cnt1;
}

static void __init irq_timings_init(void)
{
	struct clk *irq_timer_clk;

	irq_timer_base = ioremap_nocache(DMW_TIMER2_BASE, SZ_4K);

	if (!irq_timer_base)
		panic("Could not map timer(s)!");

	irq_timer_clk = clk_get_sys("timer3", NULL);

	if (IS_ERR(irq_timer_clk))
		panic("Could not get clock(s)!");

	clk_enable(irq_timer_clk);

	/* Initialize to a known state (timer off). */
	writel(0UL, irq_timer_base + DMW_TMR_CTRL);
	writel(0UL, irq_timer_base + DMW_TMR_LOAD);
	writel(0UL, irq_timer_base + DMW_TMR_VALUE);

	/* Configure the HW timer for the maximum possible frequency */
	/* Configure Timer3: free running, 32bit, Prescaler 1, Enable */
	writel(CTRL_32BIT | CTRL_ENABLE |
	       CTRL_DIV1,
	       irq_timer_base + DMW_TMR_CTRL);

	local_irq_disable();
	memset(irq_timings, 0, sizeof(irq_timings));
	local_irq_enable();

	init_irq_stack(0);	
	init_irq_stack(1);	
}

static int timproc_show(struct seq_file *s, void *v)
{
	unsigned long i;
	struct clk *irq_timer_clk;

	unsigned long max, min, irq_freq_mhz;

	irq_timer_clk = clk_get_sys("timer3", NULL);
	irq_timer_freq = clk_get_rate(irq_timer_clk);
	irq_freq_mhz = irq_timer_freq/1000000;

	printk(KERN_DEBUG"IRQ Timer clocked at %lu MHz\n",irq_freq_mhz);

	seq_printf(s, "%30s   cnt     ticks   in us:  avg  max  min \n\n","irq");

	local_irq_disable();
	for (i = 0; i < DMW_PLICU_NR_IRQS; i++) {
		unsigned long count, time, avg;

		count = irq_timings[i].count;

		if (!count)
			continue;

		time  = irq_timings[i].time / irq_freq_mhz; /*us */
		avg   = time/count; /*us*/
		min = irq_timings[i].min / irq_freq_mhz; /* us */
		max = irq_timings[i].max / irq_freq_mhz; /* us */

		seq_printf(s, "%30s  %4lu %8lu          %4lu %4lu %4lu\n",
		           irq_strings[i], count, time, avg, max, min);
	}
	local_irq_enable();

	clk_put(irq_timer_clk);

	return 0;
}

static int timproc_open(struct inode *inode, struct file *file)
{
	return single_open(file, timproc_show, NULL);
}

static const struct file_operations timproc_ops = {
	.owner = THIS_MODULE,
	.open = timproc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int __init timproc_init(void)
{
	proc_create("irq_timings", 0, NULL, &timproc_ops);
	return 0;
}
module_init(timproc_init);
#else /* CONFIG_DMW_IRQMEAS */
static void __init irq_timings_init(void)
{
}
#endif

#ifdef CONFIG_OPROFILE_USE_HWTIMER
static void *oprofile_base;

static irqreturn_t
dmw_oprofile_interrupt(int irq, void *dev_id)
{
	writel(1UL, oprofile_base + DMW_TMR_INTCLR);

	profile_tick(CPU_PROFILING);

	return IRQ_HANDLED;
}

static struct irqaction dmw_oprofile_irq = {
	.name		= "DMW oprofile timer",
	.flags		= IRQF_TIMER | IRQF_IRQPOLL,
	.handler	= dmw_oprofile_interrupt,
};

static void __init
dmw_oprofile_timer_init(void)
{
	struct clk *oprof_clk;

	oprofile_base = ioremap_nocache(DMW_TIMER1_BASE, SZ_4K);
	if (!oprofile_base)
		panic("Could not map OProfile timer!");

	oprof_clk = clk_get_sys("timer2", NULL);
	if (IS_ERR(oprof_clk))
		panic("Could not get OProfile clock!");

	setup_irq(DMW_IID_TIMER2, &dmw_oprofile_irq);

	clk_enable(oprof_clk);
	writel(0UL, oprofile_base + DMW_TMR_CTRL);
	writel(0UL, oprofile_base + DMW_TMR_LOAD);
	writel(0UL, oprofile_base + DMW_TMR_VALUE);
	/* Configure the HW timer for the maximum possible frequency */
	writel(CTRL_32BIT | CTRL_ENABLE | CTRL_PERIODIC |
	       CTRL_DIV1  | CTRL_IE,
	       oprofile_base + DMW_TMR_CTRL);
}
#else /* CONFIG_OPROFILE_USE_HWTIMER */
static void __init
dmw_oprofile_timer_init(void)
{
}
#endif

static void __init
dmw_timer_init(void)
{
	struct clk *rtc_clk;
	unsigned long rtc_freq;

	rtc_base = ioremap_nocache(DMW_RTC_BASE, SZ_4K);
	if (!rtc_base)
		panic("Could not map RTC!");

	rtc_clk = clk_get_sys("rtc", NULL);
	if (IS_ERR(rtc_clk))
		panic("Could not get RTC clock!");

	/* Reset the RTC in order to start at time zero */
	reset_set(RESET_RTC);
	/*
	 * since no *delay() call is working yet, do two reads in the pclk
	 * domain in order to make sure the reset has actually gone through
	 * to the block
	 */
	rtc_read();
	rtc_read();
	reset_release(RESET_RTC);

	/* Set up RTC */
	clk_enable(rtc_clk);
	writel(CTL_SWRST, rtc_base + DMW_RTC_CTL);
	writel(CFG_ALARMEN | CFG_INTEN, rtc_base + DMW_RTC_CFG);
	setup_irq(DMW_IID_RTC, &dmw_rtc_irq);

	/* Register RTC as clock source. */
	rtc_freq = clk_get_rate(rtc_clk);
	rtc_clocksource.read = dmw_get_cycles;
	rtc_clocksource.mult = clocksource_hz2mult(rtc_freq, rtc_clocksource.shift);
	clocksource_register(&rtc_clocksource);

	/* Register RTC as clock event device. */
	rtc_clockevent.mult = div_sc(rtc_freq, NSEC_PER_SEC, rtc_clockevent.shift);
	rtc_clockevent.max_delta_ns = clockevent_delta2ns(0xffffffff, &rtc_clockevent);
	rtc_clockevent.min_delta_ns = clockevent_delta2ns(0xf, &rtc_clockevent);
	rtc_clockevent.cpumask = cpumask_of(0);
	clockevents_register_device(&rtc_clockevent);

	/* Init other timers */
	dmw_oprofile_timer_init();
	irq_timings_init();
}

struct sys_timer dmw_timer = {
	.init = dmw_timer_init,
};

