/*
 * arch/arm/mach-dmw/css/dev-datahandler.c - DECT data handler character device
 *
 * This driver registers one character device for each available data channel.
 * It allows only one reader and one writer at the same time.
 *
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/completion.h>
#include <linux/poll.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/vmalloc.h>
#include <asm/uaccess.h>

#include "coma.h"
#include "cmsg-datahandler.h"
#include "coma-datahandler.h"
#include "coma-service.h"

MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

#define SERVICE_NAME		"coma-datahandler"
#define SERVICE_DATAHANDLER	COMA_SERVICE_DATAHANDLER
#define FIFO_SIZE		15*1024
#define DATAHANDLER_MAJOR	0

//static DECLARE_COMPLETION(datahandler_read_compl);
static DEFINE_MUTEX(datahandler_mutex);

static char datahandler_name[] = "cordless_dh";

static struct cordless_data_handler *dh;

static int datahandler_initialized = 0;

static struct cfifo *datahandler_l2c;
static struct cfifo *datahandler_c2l;

static DECLARE_COMPLETION(datahandler_reply_compl);
static DEFINE_MUTEX(coma_mutex);
static union cmsg_datahandler_params last_cmsg_datahandler_params;

#define DH_ESTABLISH_REQUESTED 1
#define DH_DISC_REQUESTED 2
#define DH_WRITE_REQUESTED 3
#define DH_INIT_REQUESTED 4
#define MAX_HANDSET 10
u8 g_dh_command_status[MAX_HANDSET + 1];

/*
 * helper functions
 */

int queryDhInstance(int handset)
{
	int i = 0;

	for (i = 0; i < MAX_HANDSET; i++) {
		if (dh->instances[i]) {
			if (dh->instances[i]->handset == handset) {
				break;
			}
		}
	}
	return i;
}
static int datahandler_registered(void)
{
	return (datahandler_initialized != 0);
}

static int datahandler_setup(void)
{
	datahandler_initialized = 1;

	return 0;
}

static void datahandler_remove(void)
{
	datahandler_initialized = 0;
}

static int
coma_create_datahandler_message(enum cmsg_datahandler_types type,
				union cmsg_datahandler_params *params,
				void *payload, unsigned int payload_size)
{
	return coma_create_message(SERVICE_DATAHANDLER, (int)type,
				   (void *)params, sizeof(*params),
				   payload, payload_size);
}

static int coma_dh_init(void)
{
	int ret;

	mutex_lock(&coma_mutex);

	if (!datahandler_registered()) {
		mutex_unlock(&coma_mutex);
		//return -EFAULT;
		return -2;
	}

	ret = coma_create_datahandler_message(CMSG_DATAHANDLER_REQUEST_INIT,
					      NULL, NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_DATAHANDLER);
#ifdef DH_DEBUG
	printk("BEFORE wait_for_completion:: coma_dh_init\n");
#endif

	wait_for_completion(&datahandler_reply_compl);
#ifdef DH_DEBUG
	printk("AFTER wait_for_completion:: coma_dh_init\n");
#endif

	ret = last_cmsg_datahandler_params.reply_init.result;

	mutex_unlock(&coma_mutex);

	return ret;
}

static int coma_dh_est_data_call(int handset, unsigned char bearertype)
{
	int ret, i;
	union cmsg_datahandler_params params;

	//mutex_lock(&coma_mutex);

	if (!datahandler_registered()) {
		//mutex_unlock(&coma_mutex);
		//return -EFAULT;
		return -2;
	}

	if ((i = queryDhInstance(handset)) == MAX_HANDSET) {
		//mutex_unlock(&coma_mutex);
		//return -EFAULT;
		return -1;
	}
	printk
	    ("Linux Driver: coma_dh_est_data_call, handset = %d, bearertype = %x\n",
	     handset, bearertype);
	dh->instances[i]->command_status = DH_ESTABLISH_REQUESTED;
	params.request_est_data_call.handset = handset;
	params.request_est_data_call.bearertype = (unsigned char)bearertype;
	mutex_lock(&coma_mutex);
	ret =
	    coma_create_datahandler_message
	    (CMSG_DATAHANDLER_REQUEST_EST_DATA_CALL, &params, NULL, 0);
	if (ret != 0) {
		mutex_unlock(&coma_mutex);
		return ret;
	} else if (ret == 0) {
#ifdef DH_DEBUG
		printk("BEFORE wait_for_completion:: coma_dh_est_data_call\n");
#endif
		coma_signal(SERVICE_DATAHANDLER);
		mutex_unlock(&coma_mutex);

		wait_event_interruptible(dh->instances[i]->dh_oper_compl,
					 (dh->instances[i]->command_status ==
					  0));
#ifdef DH_DEBUG
		printk("AFTER wait_for_completion:: coma_dh_est_data_call\n");
#endif

		ret = dh->instances[i]->dh_param.reply_est_data_call.result;

		//mutex_unlock(&coma_mutex);
	}
	return ret;
}

static int coma_dh_disc_data_call(int handset)
{
	int ret, i;
	union cmsg_datahandler_params params;

	//mutex_lock(&coma_mutex);

	if (!datahandler_registered()) {
		//mutex_unlock(&coma_mutex);
		//return -EFAULT;
		return -2;
	}
	if ((i = queryDhInstance(handset)) == MAX_HANDSET) {
		//mutex_unlock(&coma_mutex);
		return -1;
	}

	dh->instances[i]->command_status = DH_DISC_REQUESTED;
	params.request_disc_data_call.handset = handset;
	mutex_lock(&coma_mutex);
	ret =
	    coma_create_datahandler_message
	    (CMSG_DATAHANDLER_REQUEST_DISC_DATA_CALL, &params, NULL, 0);
	if (ret != 0) {
		mutex_unlock(&coma_mutex);
		return ret;
	} else if (ret == 0) {
#ifdef DH_DEBUG
		printk("BEFORE wait_for_completion:: coma_dh_disc_data_call\n");
#endif
		coma_signal(SERVICE_DATAHANDLER);

		//wait_for_completion(&datahandler_reply_compl);
		mutex_unlock(&coma_mutex);
		wait_event_interruptible(dh->instances[i]->dh_oper_compl,
					 (dh->instances[i]->command_status ==
					  0));
		//printk("AFTER wait_for_completion:: coma_dh_disc_data_call\n");
		ret = dh->instances[i]->dh_param.reply_disc_data_call.result;

		//mutex_unlock(&datahandler_mutex);

	}
	return ret;
}

static int coma_dh_write(int handset, const uint8_t * data, uint32_t count)
{
	int ret, i;
	union cmsg_datahandler_params params;

	//mutex_lock(&coma_mutex);

	if (!datahandler_registered()) {
		//mutex_unlock(&coma_mutex);
		//return -EFAULT;
		return -2;
	}

	if ((i = queryDhInstance(handset)) == MAX_HANDSET) {
		//mutex_unlock(&coma_mutex);
		//return -EFAULT;
#ifdef DH_DEBUG
		printk("No matching instance for HS %d\n", handset);
#endif
		return -1;
	}
	dh->instances[i]->command_status = DH_WRITE_REQUESTED;
	params.request_write.handset = handset;
	mutex_lock(&coma_mutex);
	ret = coma_create_datahandler_message(CMSG_DATAHANDLER_REQUEST_WRITE,
					      &params, (uint8_t *) data, count);
	if (ret != 0) {
		mutex_unlock(&coma_mutex);
#ifdef DH_DEBUG
		printk("Coma_create_datahandler_Message_failed!!!\N");
#endif
		return ret;
	} else if (ret == 0) {
#ifdef DH_DEBUG
		printk("BEFORE wait_for_completion:: coma_dh_write\n");
#endif
		coma_signal(SERVICE_DATAHANDLER);

		//wait_for_completion(&datahandler_reply_compl);
		mutex_unlock(&coma_mutex);
		wait_event_interruptible(dh->instances[i]->dh_oper_compl,
					 (dh->instances[i]->command_status ==
					  0));
#ifdef DH_DEBUG
		printk("AFTER wait_for_completion:: coma_dh_write\n");
#endif

		ret = dh->instances[i]->dh_param.reply_write.result;

		//mutex_unlock(&coma_mutex);
	}
	return ret;
}

static unsigned int datahandler_poll(struct file *file, poll_table * event_list)
{
	struct dh_instance *inst = (struct dh_instance *)file->private_data;
	unsigned int mask = 0;

	/* we always allow writing */
	mask |= POLLOUT | POLLWRNORM;

	if (inst->len)
		mask |= POLLIN | POLLRDNORM;

	poll_wait(file, &inst->dh_read_compl, event_list);
	return mask;

}

static int datahandler_open(struct inode *inode, struct file *filp)
{
	struct dh_instance *inst;
	//unsigned int flags = filp->f_flags & O_ACCMODE;
	int i;

	mutex_lock(&datahandler_mutex);

	if (!dh->open_count)
		coma_dh_init();

	if (dh->open_count >= MAX_HANDSET) {
		mutex_unlock(&datahandler_mutex);
		//return -EBUSY;
		return -1;
	}

	/* do not allow read AND write access */
	/*if (flags == O_RDWR) {
	   mutex_unlock(&datahandler_mutex);
	   return -EBUSY;
	   } */

	inst = kmalloc(sizeof(struct dh_instance), GFP_KERNEL);
	if (!inst) {
		mutex_unlock(&datahandler_mutex);
		//return -ENOMEM;
		return -8;
	}
	filp->private_data = inst;

	for (i = 0; i < MAX_HANDSET; i++) {
		if (!dh->instances[i]) {
			dh->instances[i] = inst;
			init_waitqueue_head(&(dh->instances[i]->dh_read_compl));
			init_waitqueue_head(&(dh->instances[i]->dh_oper_compl));
			break;
		}
	}
	/* upon open we don't establish the data link yet,
	 * it will be established on the first write,
	 * after ioctls may have selected another handset */
	inst->handset = -1;
	inst->established = 0;
	inst->buf = NULL;
	inst->len = 0;

	dh->open_count++;

	mutex_unlock(&datahandler_mutex);

	return 0;
}

static int dh_disc_data_call(int handset)
{
	int i;

	for (i = 0; i < MAX_HANDSET; i++) {
		if (dh->instances[i] && (dh->instances[i]->handset == handset)) {
			dh->instances[i]->established = 0;
			return 1;
		}
	}

	return 0;
}

/*This function is called when DATA CALL Active callback is invoked. In case pt initiated Data call, this function sets status as established*/
int dh_set_status_established(int handset)
{
	int i;
	for (i = 0; i < MAX_HANDSET; i++) {
		if (dh->instances[i] && (dh->instances[i]->handset == handset)) {
			if (dh->instances[i]->established != 1) {
				dh->instances[i]->established = 1;	/* Set status as 1, call established by handset */
			}
			return 1;

		}
	}

	return 0;

}
static int datahandler_release(struct inode *inode, struct file *filp)
{
	struct dh_instance *inst = filp->private_data;
	int i, ret = 0;

	mutex_lock(&datahandler_mutex);

	dh->open_count--;

	if (inst->established && (inst->handset >= 0))
		ret = coma_dh_disc_data_call(inst->handset);
//printk("Linux  Driver: Completed release of data call\n");
	inst->established = 0;
	inst->handset = -1;
	inst->bearer_type = 0x11;
	for (i = 0; i < MAX_HANDSET; i++) {
		if (dh->instances[i] == inst) {
			dh->instances[i] = 0;
			break;
		}
	}

	kfree(inst);

	mutex_unlock(&datahandler_mutex);

	return ret;
}

static int dh_est_data_call(struct dh_instance *inst)
{
	int ret = -1;
#ifdef DH_DEBUG
	printk
	    ("dh_est_data_call: inst-> handset = %d, inst->bearer_type = %x\n",
	     inst->handset, inst->bearer_type);
#endif
	ret = coma_dh_est_data_call(inst->handset, inst->bearer_type);

	return ret;
}

static ssize_t
datahandler_write(struct file *filp, const char __user * buf,
		  size_t count_want, loff_t * f_pos)
{
	int ret = 0;
	uint8_t *lbuf;
	struct dh_instance *inst = filp->private_data;

	if (!datahandler_registered())
		//return -EFAULT;
		return -2;
#ifdef DH_DEBUG
	if (count_want > 3500)
		return -1;
#else
	if (count_want > 1500)
		return -1;
#endif

/*	if (!cordless_initialized())
		//return -EFAULT;
		return -2;
*/
	if (inst->handset < 0) {
		//return -EFAULT;
		return -1;
	}

	if (inst->established == 0) {
		mutex_lock(&datahandler_mutex);
		ret = dh_est_data_call(inst);
#ifdef DH_DEBUG
		printk
		    ("Linux  Driver: Completed establishment of data call(coma_dh_est_data_call returned %d)\n",
		     ret);
#endif
		mutex_unlock(&datahandler_mutex);
		if (ret < 0) {
			inst->established = 0;
			//return -EFAULT;
			return ret;
		}
		//inst->established = 1;
	}

	lbuf = vmalloc(count_want);
	if (!lbuf) {
		//return -ENOMEM;
		return -8;
	}

	ret = copy_from_user(lbuf, buf, count_want);
	if (ret != 0) {
		vfree(lbuf);
		//return -EFAULT;
		return -6;
	}
	//printk("Inside cordless_dh_write: before calling cordless_dh_write\n");
	/* we only copy the data pointer and size to the cordless stack,
	 * this works as we're both in kernel mode */
	/*This mutex lock below is commented for O_RDWR support */
	//mutex_lock(&dh_mutex);
#ifdef DH_DEBUG
	printk("Linux  Driver: started coma_dh_write, for hs %d\n",
	       inst->handset);
#endif
	ret = coma_dh_write(inst->handset, lbuf, count_want);
#ifdef DH_DEBUG
	printk("Linux  Driver: Completed coma_dh_write for hs %d\n",
	       inst->handset);
#endif
	//mutex_unlock(&dh_mutex);
	vfree(lbuf);
	if (ret != 0)
		//return -EFAULT;
		return ret;

	return count_want;
}
void datahandler_data(int handset, int result, unsigned char *buf, int len)
{
	int i;
	//printk("datahandler_data: handset = %d, length = %d\n",handset,len);
	//mutex_lock(&datahandler_mutex);
	for (i = 0; i < MAX_HANDSET; i++) {
		if (dh->instances[i]) {
			if (dh->instances[i]->handset == handset) {
				dh->instances[i]->buf = buf;
				dh->instances[i]->len = len;
				wake_up(&dh->instances[i]->dh_read_compl);
				break;
			}
		}
	}
	//mutex_unlock(&datahandler_mutex);
	//printk("Linux  Driver: Received DH data %d\n", len);
}

static int datahandler_process_message(struct cmsg *cmsg)
{
	int ret = 0, i;
	union cmsg_datahandler_params *params =
	    (union cmsg_datahandler_params *)(cmsg->params);

	switch (cmsg->type) {
	case CMSG_DATAHANDLER_REPLY_READ:
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_REPLY_DH_READ\n");
#endif
		datahandler_data(params->reply_read.handset,
				 params->reply_read.result,
				 cmsg->payload, cmsg->payload_size);
		break;
	case CMSG_DATAHANDLER_REPLY_DISC_DATA_CALL:
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_REPLY_DH_DISC_DATA_CALL\n");
#endif
		dh_disc_data_call(params->reply_disc_data_call.handset);
		if ((i =
		     queryDhInstance(params->reply_disc_data_call.handset)) ==
		    MAX_HANDSET) {
#ifdef DH_DEBUG
			printk
			    ("CMSG_DATAHANDLER_REPLY_DISC_DATA_CALL: invalid handset number received\n");
#endif
			break;
		}
		if (dh->instances[i]->command_status == DH_DISC_REQUESTED) {
			//printk("dh->instances[%d]->command_status freed from DH_DISC_REQUESTED . Handset is %d\n",i, dh->instances[i]->handset);
			dh->instances[i]->command_status = 0;
			//dh_params= *params;
			dh->instances[i]->dh_param = *params;
			wake_up(&dh->instances[i]->dh_oper_compl);
		}
		break;
	case CMSG_DATAHANDLER_HS_DISC_DATA_CALL:
		dh_disc_data_call(params->reply_disc_data_call.handset);
#ifdef DH_DEBUG
		printk
		    ("Coma Process: Got CMSG_DATAHANDLER_HS_DISC_DATA_CALL\n");
#endif
		if ((i =
		     queryDhInstance(params->reply_disc_data_call.handset)) ==
		    MAX_HANDSET) {
			printk
			    ("CMSG_DATAHANDLER_HS_DISC_DATA_CALL: invalid handset number received\n");
			break;
		}

		if (dh->instances[i]->command_status == DH_ESTABLISH_REQUESTED) {
			//printk("dh->instances[%d]->command_status freed from DH_ESTABLISH_REQUESTED . Handset is %d\n",i, dh->instances[i]->handset);
			dh->instances[i]->command_status = 0;
			dh->instances[i]->dh_param = *params;
			wake_up(&dh->instances[i]->dh_oper_compl);
		} else if (dh->instances[i]->command_status ==
			   DH_WRITE_REQUESTED) {
			//printk("dh->instances[%d]->command_status freed from DH_WRITE_REQUESTED . Handset is %d\n",i, dh->instances[i]->handset);
			dh->instances[i]->command_status = 0;
			dh->instances[i]->dh_param = *params;
			wake_up(&dh->instances[i]->dh_oper_compl);

		}
		break;
	case CMSG_DATAHANDLER_REPLY_EST_DATA_CALL:
		dh_set_status_established(params->reply_est_data_call.handset);
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_REPLY_DH_EST_DATA_CALL\n");
#endif
		if ((i =
		     queryDhInstance(params->reply_disc_data_call.handset)) ==
		    MAX_HANDSET) {
			printk
			    ("CMSG_REPLY_DH_EST_DATA_CALL: invalid handset number received\n");
			break;
		}

		if (dh->instances[i]->command_status == DH_ESTABLISH_REQUESTED) {
			//printk("dh->instances[%d]->command_status freed from DH_ESTABLISH_REQUESTED . Handset is %d\n",i, dh->instances[i]->handset);
			dh->instances[i]->command_status = 0;
			dh->instances[i]->dh_param = *params;
			wake_up(&dh->instances[i]->dh_oper_compl);

		}
		break;
	case CMSG_DATAHANDLER_REPLY_WRITE:
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_DATAHANDLER_REPLY_WRITE\n");
#endif
		if ((i =
		     queryDhInstance(params->reply_write.handset)) ==
		    MAX_HANDSET) {
			printk
			    ("CMSG_DATAHANDLER_REPLY_WRITE: invalid handset number received\n");
			break;
		}

		if (dh->instances[i]->command_status == DH_WRITE_REQUESTED) {
			//printk("dh->instances[%d]->command_status freed from DH_WRITE_REQUESTED . Handset is %d\n",i, dh->instances[i]->handset);
			dh->instances[i]->command_status = 0;
			dh->instances[i]->dh_param = *params;
			wake_up(&dh->instances[i]->dh_oper_compl);
		}
		break;
	case CMSG_DATAHANDLER_REPLY_INIT:
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_DATAHANDLER_REPLY_INIT\n");
#endif
		last_cmsg_datahandler_params = *params;
		complete(&datahandler_reply_compl);
		break;
	case CMSG_DATAHANDLER_HS_EST_DATA_CALL:
#ifdef DH_DEBUG
		printk("Coma Process: Got CMSG_DATAHANDLER_HS_EST_DATA_CALL\n");
#endif
		dh_set_status_established(params->reply_est_data_call.handset);
		break;
		/* DATA HANDLER MESSAGES END */
//#endif /* CONFIG_CORDLESS_DH */
		//case CMSG_REPLY_START_PCM_CHAN:
		//case CMSG_REPLY_STOP_PCM_CHAN:
		//      coma_generic_reply(params);
		break;
	default:
		ret = -1;
		break;
	}

	return ret;
}

static ssize_t
datahandler_read(struct file *filp, char __user * buf,
		 size_t count_want, loff_t * f_pos)
{
	int ret = 0;
	struct dh_instance *inst = filp->private_data;

	if (!datahandler_registered())
		//return -EFAULT;
		//if (!cordless_initialized())
		return -2;
	if (inst->handset < 0)
		//return -EFAULT;
		return -1;

	//ret = wait_for_completion_interruptible(&datahandler_read_compl);
	ret = wait_event_interruptible(inst->dh_read_compl, (inst->len > 0));
	if (ret == -ERESTARTSYS) {
		printk("datahandler_read: ret = -ERESTARTSYS\n");
		inst->buf = NULL;
		inst->len = 0;
		//return ret;
		return -5;
	}

	if ((inst->buf != NULL) && (inst->len != 0)) {
		if (inst->len > count_want)
			inst->len = count_want;
		if (copy_to_user(buf, inst->buf, inst->len)) {
			inst->buf = NULL;
			inst->len = 0;
			printk("datahandler_read: return -1\n");
			//return -EFAULT;
			return -1;
		}

		ret = inst->len;
		inst->buf = NULL;
		inst->len = 0;
	} else {
		printk("datahandler_read: return -1\n");
		ret = -1;
		return ret;
	}

	return ret;
}

static long
datahandler_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct dh_instance *inst = filp->private_data;
	PDATACALL_INFO pcall_info = (PDATACALL_INFO) (arg);
	int ret = 0;
	if (!datahandler_registered())
		//return -1;
		return -2;

/*	if (!cordless_initialized())
	{
		printk("Cordless not initialized!\n");
		return -2;
	}
*/
	switch (cmd) {
	case IOCTL_CORDLESS_DH_SET_HANDSET_AND_BEARERTYPE:
		if (inst->established && (inst->handset >= 0)) {
			printk(KERN_ERR "datahandler_ioctl() error: "
			       "cannot switch handset on established"
			       "datacalls\n");
			return -1;
		}
		inst->handset = pcall_info->handset;
		inst->bearer_type = pcall_info->bearertype;
#ifdef DH_DEBUG
		printk("cordless_dh_ioctl() : bearer type set to  %x\n",
		       inst->bearer_type);
#endif
		break;
	case IOCTL_CORDLESS_DH_DISCONNECT_AIRLINK:
		if (inst->established && (inst->handset >= 0)) {
			mutex_lock(&datahandler_mutex);
			ret = coma_dh_disc_data_call(inst->handset);
			mutex_unlock(&datahandler_mutex);
		} else {
			printk
			    ("cordless_dh_ioctl() error:cannot disconnect handset as already disconnected \n");
			return -9;
		}
		break;
/*		case IOCTL_CORDLESS_DH_GET_HANDSET:
			inst->handset = arg;
			break;
*/

	default:
		printk(KERN_ERR "ioctl not implemented\n");
		ret = -1;
	}
	//return 0;
	return ret;
}

static struct file_operations datahandler_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl	= datahandler_ioctl,
	.read		= datahandler_read,
	.open		= datahandler_open,
	.write		= datahandler_write,
	.release	= datahandler_release,
	.poll		= datahandler_poll,
};

/*
 * module operations
 */

static int cfifo_size;
static dma_addr_t cfifo_phys;

int datahandler_init(void)
{
	int i;
	int ret = 0;

	dh = kmalloc(sizeof(struct cordless_data_handler), GFP_KERNEL);
	if (!dh)
		return -ENOMEM;

	dh->open_count = 0;
	for (i = 0; i < MAX_HANDSET; i++)
		dh->instances[i] = 0;

	if (DATAHANDLER_MAJOR) {
		dh->dev = MKDEV(DATAHANDLER_MAJOR, 0);
		ret =
		    register_chrdev_region(dh->dev, MAX_HANDSET,
					   datahandler_name);
	} else
		ret =
		    alloc_chrdev_region(&dh->dev, 0, MAX_HANDSET,
					datahandler_name);

/*ret = alloc_chrdev_region(&dh->dev, 0, MAX_HANDSET, datahandler_name);*/
	if (ret) {

		printk(KERN_ERR
		       "character device registration failed ret = %x\n", ret);

		goto err_kfree;
	}

	cdev_init(&dh->cdev, &datahandler_fops);

	dh->cdev.owner = THIS_MODULE;
	dh->cdev.ops = &datahandler_fops;

//      ret = cdev_add(&dh->cdev, dh->dev, 1);
	ret = cdev_add(&dh->cdev, dh->dev, MAX_HANDSET);
	if (ret) {
		printk(KERN_ERR "%s: Error %d adding character device",
		       datahandler_name, ret);
		goto err_unreg_chrdev_reg;
	}

	cfifo_size =
	    cfifo_alloc(FIFO_SIZE, &datahandler_c2l, FIFO_SIZE,
			&datahandler_l2c, &cfifo_phys);
	if (cfifo_size < 0) {
		ret = -ENOMEM;
		goto err_cdev_del;
	}

	ret = coma_register(SERVICE_DATAHANDLER, SERVICE_NAME, datahandler_l2c,
			    datahandler_c2l, datahandler_process_message,
			    datahandler_setup, datahandler_remove);
	if (ret < 0) {
		printk(KERN_ERR "%s: Registration failed: %d\n",
		       datahandler_name, ret);
		goto err_cfifo_free;
	}

	printk(KERN_INFO "%s: character device initialized (major=%d)\n",
	       datahandler_name, MAJOR(dh->dev));

#if 0
	spin_lock_init(&log_messages_lock);
	log_messages = kfifo_alloc(4096, GFP_KERNEL, &log_messages_lock);
#endif

	return 0;

      err_cfifo_free:
	cfifo_free((char *)datahandler_c2l, cfifo_size, cfifo_phys);
      err_cdev_del:
	cdev_del(&dh->cdev);
      err_unreg_chrdev_reg:
	//unregister_chrdev_region(dh->dev, 1);
	unregister_chrdev_region(dh->dev, MAX_HANDSET);
      err_kfree:
	kfree(dh);

	return ret;
}

void datahandler_exit(void)
{
	coma_deregister(SERVICE_DATAHANDLER);
	cfifo_free((char *)datahandler_c2l, cfifo_size, cfifo_phys);
	cdev_del(&dh->cdev);
	unregister_chrdev_region(dh->dev, MAX_HANDSET);
	kfree(dh);
}

EXPORT_SYMBOL(datahandler_init);
EXPORT_SYMBOL(datahandler_exit);
