/*
 * arch/arm/mach-dmw/dbm.c - the DRAM Bus Monitor character device driver
 *
 * This driver registers the character device for the DRAM Controller
 * monitoring.
 *
 * Copyright (C) 2011 DSPG Technologies
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/clk.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <mach/platform.h>
#include <mach/hardware.h>

#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#define DRIVER_NAME "dbm"

#define DBM_MATCH_RDY_BIT		(1 << 31)   /* RO */
#define DBM_PERF_RDY_BIT		(1 << 30)   /* RO */
#define DBM_MATCH_EN_BIT		(1 << 1)    /* RW */
#define DBM_PERF_EN_BIT			 1          /* RW */

#define DBM_DRAM_PERF_EN_BIT		(1 << 8)      /* RW */
#define DBM_PORT_PERF_EN_BIT(PORT)	(1 << (PORT)) /* RW */

#define DBM_PERF_DIR_SHIFT(PORT)	((PORT) << 2)

#define DMW_DBM_PERF_START_MASK		0x1
#define DMW_DBM_PERF_STOP_MASK		0x1

#define DBM_PORT_MATCH_EN_BIT(PORT)	(1 << (PORT)) /* RW */

#define DBM_DRAM_PERF_ROV_IND_BIT	(1 << 8)      /* RO */
#define DBM_P_PERF_ROV_IND_BIT(PORT)	(1 << (PORT)) /* RO */

#define DBM_DRAM_PERF_ROV_IND_CLR_BIT	(1 << 8)      /* WO */
#define DBM_P_PERF_ROV_IND_CLR_BIT(PORT)	(1 << (PORT)) /* WO */

static u32 dbm_mem;
static struct clk *clk_dbm;

static irqreturn_t
dbm_isr(int irq, void *dev_id)
{
	unsigned int handled = 0;

	return IRQ_RETVAL(handled);
}

static inline u32
dbm_read(u32 addr)
{
	return readl(dbm_mem + addr);
}

static inline void
dbm_write(u32 addr, u32 val)
{
	writel(val, dbm_mem + addr);
}

static inline void
dbm_hw_perf_enable(u8 enable)
{
	u32 val = dbm_read(DMW_DBM_CONFIG);

	val = enable ? (val | DBM_PERF_EN_BIT) : (val & ~(DBM_PERF_EN_BIT));

	dbm_write(DMW_DBM_CONFIG, val);
}

static inline void
dbm_hw_match_enable(u8 enable)
{
	u32 val = dbm_read(DMW_DBM_CONFIG);

	val = enable ? (val | DBM_MATCH_EN_BIT) : (val & ~(DBM_MATCH_EN_BIT));

	dbm_write(DMW_DBM_CONFIG, val);
}

static inline u8
dbm_hw_check_match_ready(void)
{
	return (dbm_read(DMW_DBM_CONFIG) & DBM_MATCH_RDY_BIT);
}

static inline u8
dbm_hw_check_perf_ready(void)
{
	return (dbm_read(DMW_DBM_CONFIG) & DBM_PERF_RDY_BIT);
}

static inline void
dbm_hw_dram_perf_enable(u8 enable)
{
	u32 val = dbm_read(DMW_DBM_PERF_CONFIG);

	val = enable ? (val | DBM_DRAM_PERF_EN_BIT) : (val & ~DBM_DRAM_PERF_EN_BIT);

	dbm_write(DMW_DBM_PERF_CONFIG, val);
}

/*
 * enable performance counter for a particular port
 * port - port number, starting from 0, -1 for all
 */
static inline void
dbm_hw_port_perf_enable(u8 port, u8 enable)
{
	u32 val = dbm_read(DMW_DBM_PERF_CONFIG);

	if (port == (u8)(-1))
		val = enable ? (val | 0xFF) : (val & ~0xFF);
	else
		val = enable ? (val | DBM_PORT_PERF_EN_BIT(port)) : (val & ~DBM_PORT_PERF_EN_BIT(port));

	dbm_write(DMW_DBM_PERF_CONFIG, val);
}

static inline void
dbm_hw_port_match_enable(u8 port, u8 enable)
{
	u32 val;

	val = dbm_read(DMW_DBM_MATCH_CONFIG);

	if (port == (u8)(-1)) {
		val = enable ? (val | 0xFF) : (val & ~0xFF);
	} else {
		val = enable ? (val | DBM_PORT_PERF_EN_BIT(port)) : (val & ~DBM_PORT_PERF_EN_BIT(port));
	}

	dbm_write(DMW_DBM_MATCH_CONFIG, val);
}

static inline void
dbm_hw_write_port_perf_dir(u8 port, u8 dir)
{
	u32 val;

	val = dbm_read(DMW_DBM_PERF_DIR);

	val |= dir << DBM_PERF_DIR_SHIFT(port);

	dbm_write(DMW_DBM_PERF_DIR, val);
}

static inline u8
dbm_hw_read_port_perf_dir(u8 port)
{
	return ( (dbm_read(DMW_DBM_PERF_DIR) & (0xFF << DBM_PERF_DIR_SHIFT(port))) >> DBM_PERF_DIR_SHIFT(port) );
}

static inline void
dbm_hw_perf_start(void)
{
	dbm_write(DMW_DBM_PERF_START, DMW_DBM_PERF_START_MASK);
}

static inline void
dbm_hw_perf_stop(void)
{
	dbm_write(DMW_DBM_PERF_STOP, DMW_DBM_PERF_STOP_MASK);
}

static inline u8
dbm_hw_check_dram_rov(void)
{
	return (dbm_read(DMW_DBM_STATUS) & DBM_DRAM_PERF_ROV_IND_BIT);
}

static inline u8
dbm_hw_check_port_rov(u8 port)
{
	return (dbm_read(DMW_DBM_STATUS) & DBM_P_PERF_ROV_IND_BIT(port));
}

static inline void
dbm_hw_check_dram_clear(void)
{
	dbm_write(DMW_DBM_STATUS_CLR, DBM_DRAM_PERF_ROV_IND_CLR_BIT);
}

static inline void
dbm_hw_check_port_clear(u8 port)
{
	dbm_write(DMW_DBM_STATUS_CLR, DBM_P_PERF_ROV_IND_CLR_BIT(port));
}

/* proc fs */
static int
read_proc_stat(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;

	dbm_hw_perf_enable(0);

	len += sprintf(page + len, "port0   %u\n", dbm_read(DMW_DBM_P0_PERF_CNT));
	len += sprintf(page + len, "port1   %u\n", dbm_read(DMW_DBM_P1_PERF_CNT));
	len += sprintf(page + len, "port2   %u\n", dbm_read(DMW_DBM_P2_PERF_CNT));
	len += sprintf(page + len, "port3   %u\n", dbm_read(DMW_DBM_P3_PERF_CNT));
	len += sprintf(page + len, "port4   %u\n", dbm_read(DMW_DBM_P4_PERF_CNT));
	len += sprintf(page + len, "port5   %u\n", dbm_read(DMW_DBM_P5_PERF_CNT));
	len += sprintf(page + len, "port6   %u\n", dbm_read(DMW_DBM_P6_PERF_CNT));
	len += sprintf(page + len, "port7   %u\n", dbm_read(DMW_DBM_P7_PERF_CNT));
	len += sprintf(page + len, "dram    %u\n", dbm_read(DMW_DBM_DRAM_PERF_CNT));

	dbm_hw_perf_enable(1);

	return len;
}

static int
write_proc_reset(struct file *file, const char *buffer, unsigned long count, void *data)
{
	int val;

	if (!count)
		return -EFAULT;

	get_user(val, buffer);

	if (val != (int)'1')
		return -EFAULT;

	dbm_hw_perf_stop();
	dbm_hw_perf_start();

	return count;
}

static int
dbm_probe(struct platform_device *pdev)
{
	int irq, ret = 0;
	static struct proc_dir_entry *dbm_pentry, *pentry;
	struct resource *res;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -EINVAL;

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	if (!request_mem_region(res->start, res->end - res->start + 1,
	    pdev->name)) {
		dev_err(&pdev->dev, "failed to request memory resource\n");
		return -EBUSY;
		//goto out_kfree;
	}

	dbm_mem = (u32)ioremap_nocache(res->start, res->end - res->start + 1);
	if (!dbm_mem) {
		dev_err(&pdev->dev, "failed to map registers\n");
		release_mem_region(res->start, res->end - res->start + 1);
		return -ENOMEM;
	}

	/* dbm clock enable */
	clk_dbm = clk_get(&pdev->dev, DRIVER_NAME);
	if (!clk_dbm) {
		dev_err(&pdev->dev, "cannot obtain clock");
		return -ENOENT;
	}
	clk_enable(clk_dbm);

	/* request interrupt for counters roll over */
	ret = request_irq(irq, dbm_isr, IRQF_SHARED, DRIVER_NAME, pdev);
	if (ret < 0) {
		dev_err(&pdev->dev, "cannot request interrupt");
		return ret;
	}

	dbm_pentry = proc_mkdir(DRIVER_NAME, 0);
	if (!dbm_pentry)
		return -EFAULT;

	pentry = create_proc_read_entry("stat", 0444, dbm_pentry, read_proc_stat, 0);
	if (!pentry) {
		remove_proc_entry(DRIVER_NAME, 0);
		goto out_irq;
	}

	pentry = create_proc_entry("reset", 0222, dbm_pentry);
	if (!pentry) {
		remove_proc_entry("stat", dbm_pentry);
		remove_proc_entry(DRIVER_NAME, 0);
		goto out_irq;
	}

	pentry->write_proc = write_proc_reset;

	//hw_reset();

	dbm_hw_dram_perf_enable(1);

	dbm_hw_port_perf_enable((u8)-1, 1);

	//dbm_hw_port_match_enable((u8)-1, 1);    /* for interrupt only */

	//dbm_hw_match_enable(1);

	dbm_hw_perf_enable(1);

	dbm_hw_perf_start();

	dev_info(&pdev->dev, "DBM driver registered");

	return 0;

out_irq:
	free_irq(irq, (void *)pdev);

	return ret;
}

static int dbm_suspend(struct device *dev)
{
	dbm_hw_perf_enable(0);
	clk_disable(clk_dbm);

	return 0;
}

static int dbm_resume(struct device *dev)
{
	clk_enable(clk_dbm);
	dbm_hw_perf_enable(1);

	return 0;
}

static struct dev_pm_ops dbm_pm_ops = {
	.suspend = dbm_suspend,
	.resume = dbm_resume,
};

struct platform_driver dbm_driver = {
	.driver = {
		.name  = DRIVER_NAME,
		.owner = THIS_MODULE,
		.pm = &dbm_pm_ops,
	},
};

static int __init
dbm_init(void)
{
	return platform_driver_probe(&dbm_driver, dbm_probe);
}

module_init(dbm_init);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("DMW96 Debug Bus Monitor driver");
MODULE_AUTHOR("Vadim Goldberg <vadim.goldberg@dspg.com>");
