/*
 * Provides I2C support for Philips PNX010x/PNX4008 boards.
 *
 * Authors: Dennis Kovalev <dkovalev@ru.mvista.com>
 *	    Vitaly Wool <vwool@ru.mvista.com>
 *
 * 2004-2006 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/timer.h>
#include <linux/completion.h>
#include <linux/platform_device.h>
#include <linux/i2c-pnx.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/gpio.h>

#include <mach/hardware.h>
#include <mach/i2c.h>

#define I2C_PNX_TIMEOUT		100 /* msec */
#define I2C_PNX_SPEED_KHZ	100
#define I2C_PNX_REGION_SIZE	0x100
#define I2C_PNX_FIFO_DEPTH	64

struct i2c_pnx_mif {
	int			ret;		/* Return value */
	int			mode;		/* Interface mode */
	struct completion	complete;	/* I/O completion */
	struct timer_list	timer;		/* Timeout */
	u8 *			buf;		/* Data buffer */
	int			len;		/* Length of data buffer */
	int			dummy;		/* Number of dummy bytes sent */
};

struct i2c_pnx_algo_data {
	void __iomem		*ioaddr;
	struct i2c_pnx_mif	mif;
	int			last;
	struct clk		*clk;
	struct i2c_pnx_data	*i2c_pnx;
	struct i2c_adapter	adapter;
};

/**
 * i2c_pnx_stop - stop a device
 * @adap:		pointer to I2C adapter structure
 *
 * Generate a STOP signal to terminate the master transaction.
 */
static void i2c_pnx_stop(struct i2c_pnx_algo_data *alg_data)
{
	/* Only 1 msec max timeout due to interrupt context */
	long timeout = 1000;

	dev_dbg(&alg_data->adapter.dev, "%s(): entering: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));

	/* Write a STOP bit to TX FIFO */
	iowrite32(0xff | stop_bit, I2C_REG_TX(alg_data));

	/* Wait until the STOP is seen. */
	while (timeout > 0 &&
	       (ioread32(I2C_REG_STS(alg_data)) & mstatus_active)) {
		/* may be called from interrupt context */
		udelay(1);
		timeout--;
	}

	dev_dbg(&alg_data->adapter.dev, "%s(): exiting: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));
}

static void i2c_pnx_master_xmit(struct i2c_pnx_algo_data *alg_data, u32 stat)
{
	int index;
	u32 val;

	dev_dbg(&alg_data->adapter.dev, "%s(): entering: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));

	/* Check if transmission is finished */
	if (stat & mstatus_tdi) {
		iowrite32(mstatus_tdi, I2C_REG_STS(alg_data));

		if (alg_data->mif.len != 0) {
			dev_err(&alg_data->adapter.dev, "TDI error\n");
			alg_data->mif.ret = -EIO;
		}
		del_timer_sync(&alg_data->mif.timer);
		complete(&alg_data->mif.complete);
	}

	/* check for drmi interrupt to ensure that SCL is hold down and fifo is empty */
	if (stat & mstatus_drmi) {
		index = 0;

		dev_dbg(&alg_data->adapter.dev, "%s(): drmi: len=%d\n", __func__, alg_data->mif.len);

		/* fill FIFO completely */
		while ((alg_data->mif.len != 0) && (index < I2C_PNX_FIFO_DEPTH)) {
			val = *alg_data->mif.buf++;

			alg_data->mif.len--;
			index++;

			if (alg_data->mif.len == 0) {
				if (alg_data->last)
					val |= stop_bit;
			}

			iowrite32(val, I2C_REG_TX(alg_data));
			dev_dbg(&alg_data->adapter.dev, "%s(): writing 0x%x.\n",
				__func__, val);
		}
	}

	if (!alg_data->last && (alg_data->mif.len == 0)) {
		iowrite32(ioread32(I2C_REG_CTL(alg_data)) & ~mcntrl_drmie,
			  I2C_REG_CTL(alg_data));

		del_timer_sync(&alg_data->mif.timer);
		complete(&alg_data->mif.complete);
	}

	dev_dbg(&alg_data->adapter.dev, "%s(): exiting: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));
}

static void i2c_pnx_master_rcv(struct i2c_pnx_algo_data *alg_data, u32 stat)
{
	u32 val = 0;
	int index;

	dev_dbg(&alg_data->adapter.dev, "%s(): entering: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));

	/* if it is the first transfer fill only the Tx FIFO */
	if ((stat & mstatus_rfe) && !(stat & mstatus_tdi)) {
		index = 0;
		while (!(ioread32(I2C_REG_STS(alg_data)) & mstatus_tff) &&
		        (index < I2C_PNX_FIFO_DEPTH)) {
			index++;

			if (alg_data->mif.dummy == 1)
				val |= stop_bit;
			if (alg_data->mif.dummy > 0) {
				/* Write dummy data to Tx FIFO */
				iowrite32(val, I2C_REG_TX(alg_data));
				alg_data->mif.dummy--;
			} else
				break;
		}
	} else {
		/* Check if transmission is finished */
		if (stat & mstatus_tdi) {
			iowrite32(mstatus_tdi, I2C_REG_STS(alg_data));

			while (alg_data->mif.len > 0) {
				if (ioread32(I2C_REG_STS(alg_data)) & mstatus_rfe) {
					alg_data->mif.ret = -EIO;
					dev_dbg(&alg_data->adapter.dev, "%s(): Transaction done but still data in the FIFO\n", __func__);
					del_timer_sync(&alg_data->mif.timer);
					complete(&alg_data->mif.complete);
					break;
				} else {
					val = ioread32(I2C_REG_RX(alg_data));
					*alg_data->mif.buf++ = (u8) (val & 0xff);
					alg_data->mif.len--;
					dev_dbg(&alg_data->adapter.dev, "%s(): reading 0x%x.\n",
						__func__, val);
				}
			}
			if (!(ioread32(I2C_REG_STS(alg_data)) & mstatus_rfe)) {
				/* FIFO throw away */
				while (!(ioread32(I2C_REG_STS(alg_data)) & mstatus_rfe))
					val = ioread32(I2C_REG_RX(alg_data));
			}
			del_timer_sync(&alg_data->mif.timer);
			complete(&alg_data->mif.complete);
		} else {
			/* Check the receiver FIFO */
			index = 0;
			while ((alg_data->mif.len > 0) && (index < I2C_PNX_FIFO_DEPTH) &&
			       !(ioread32(I2C_REG_STS(alg_data)) & mstatus_rfe)) {
				/* Read data from  Rx FIFO */
				val = ioread32(I2C_REG_RX(alg_data));
				*alg_data->mif.buf++ = (u8) (val & 0xff);
				index++;
				alg_data->mif.len--;
				dev_dbg(&alg_data->adapter.dev, "%s(): reading 0x%x.\n",
					__func__, val);
			}

			/* if there are still data to be received, fill the Tx FIFO */
			index = 0;
			while (!(ioread32(I2C_REG_STS(alg_data)) & mstatus_tff) &&
			        (index < I2C_PNX_FIFO_DEPTH)) {
				index++;
				if (alg_data->mif.dummy == 1)
					val |= stop_bit;
				if (alg_data->mif.dummy > 0) {
					/* Write dummy data to Tx FIFO    */
					iowrite32(val, I2C_REG_TX(alg_data));
					alg_data->mif.dummy--;
				} else
					break;
			}
		}
	}

	dev_dbg(&alg_data->adapter.dev, "%s(): exiting: stat = %04x.\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));
}

static irqreturn_t i2c_pnx_interrupt(int irq, void *dev_id)
{
	struct i2c_pnx_algo_data *alg_data = dev_id;
	u32 stat;

	dev_dbg(&alg_data->adapter.dev,
		"%s(): mstat = %x mctrl = %x, mode = %d\n",
		__func__,
		ioread32(I2C_REG_STS(alg_data)),
		ioread32(I2C_REG_CTL(alg_data)),
		alg_data->mif.mode);
	stat = ioread32(I2C_REG_STS(alg_data));

	/* check for error */
	if (stat & mstatus_afi) {
		dev_err(&alg_data->adapter.dev,
			"%s: Arbitration failure.\n",
			alg_data->adapter.name);
		alg_data->mif.ret = -EIO;
	} else if (stat & mstatus_nai) {
		dev_dbg(&alg_data->adapter.dev,
			"%s: Slave did not acknowledge.\n",
			alg_data->adapter.name);
		//i2c_pnx_stop(alg_data);
		alg_data->mif.ret = -EIO;
	} else {
		if (alg_data->mif.mode == I2C_SMBUS_WRITE)
			i2c_pnx_master_xmit(alg_data, stat);
		else if (alg_data->mif.mode == I2C_SMBUS_READ)
			i2c_pnx_master_rcv(alg_data, stat);
	}

	/* hardware error handling */
	if (alg_data->mif.ret < 0) {
		if (stat & mstatus_afi)
			iowrite32(mstatus_afi, I2C_REG_STS(alg_data));
		if (stat & mstatus_nai)
			iowrite32(ioread32(I2C_REG_CTL(alg_data)) & ~(mcntrl_naie | mcntrl_drmie),
				  I2C_REG_CTL(alg_data));
		if (stat & mstatus_tdi)
			iowrite32(mstatus_tdi, I2C_REG_STS(alg_data));
		del_timer_sync(&alg_data->mif.timer);
		complete(&alg_data->mif.complete);
	}

	dev_dbg(&alg_data->adapter.dev,
		"%s(): exiting, stat = %x ctrl = %x.\n",
		 __func__, ioread32(I2C_REG_STS(alg_data)),
		 ioread32(I2C_REG_CTL(alg_data)));

	return IRQ_HANDLED;
}

static inline void i2c_pnx_arm_timer(struct i2c_pnx_algo_data *alg_data)
{
	struct timer_list *timer = &alg_data->mif.timer;
	unsigned long expires = msecs_to_jiffies(I2C_PNX_TIMEOUT);

	if (expires <= 1)
		expires = 2;

	del_timer_sync(timer);

	dev_dbg(&alg_data->adapter.dev, "Timer armed at %lu plus %lu jiffies.\n",
		jiffies, expires);

	timer->expires = jiffies + expires;
	timer->data = (unsigned long)alg_data;

	add_timer(timer);
}

static int i2c_pnx_start(unsigned char slave_addr,
	struct i2c_pnx_algo_data *alg_data)
{
	volatile u32 val;

	dev_dbg(&alg_data->adapter.dev, "%s(): addr 0x%x mode %d\n", __func__,
		slave_addr, alg_data->mif.mode);

	/* Check for 7 bit slave addresses only */
	if (slave_addr & ~0x7f) {
		dev_err(&alg_data->adapter.dev,
			"%s: Invalid slave address %x. Only 7-bit addresses are supported\n",
			alg_data->adapter.name, slave_addr);
		return -EINVAL;
	}

	dev_dbg(&alg_data->adapter.dev, "%s(): sending %#x\n", __func__,
		(slave_addr << 1) | start_bit | alg_data->mif.mode);

	val = ioread32(I2C_REG_STS(alg_data));

	/* Write the slave address, START bit and R/W bit */
	iowrite32((slave_addr << 1) | start_bit | alg_data->mif.mode,
		  I2C_REG_TX(alg_data));

	/* if (len==1 && WRITE_THEN_READ) add "| stop_bit" */

	iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_naie,
		  I2C_REG_CTL(alg_data));

	val = ioread32(I2C_REG_STS(alg_data));

	if (alg_data->mif.mode == I2C_SMBUS_WRITE)
		i2c_pnx_master_xmit(alg_data, mstatus_drmi);
	else if (alg_data->mif.mode == I2C_SMBUS_READ)
		i2c_pnx_master_rcv(alg_data, mstatus_rfe);

	dev_dbg(&alg_data->adapter.dev, "%s(): exit\n", __func__);

	return 0;
}

static inline int wait_reset(long timeout, struct i2c_pnx_algo_data *data)
{
	while (timeout > 0 &&
			(ioread32(I2C_REG_CTL(data)) & mcntrl_reset)) {
		mdelay(1);
		timeout--;
	}
	return (timeout <= 0);
}

static void i2c_pnx_recover(struct i2c_pnx_algo_data *alg_data)
{
	if (alg_data->i2c_pnx->recover)
		alg_data->i2c_pnx->recover(alg_data->i2c_pnx->scl,
					   alg_data->i2c_pnx->sda);
}

static void i2c_pnx_timeout(unsigned long data)
{
	struct i2c_pnx_algo_data *alg_data = (struct i2c_pnx_algo_data *)data;
	u32 ctl;

	dev_err(&alg_data->adapter.dev,
		"Master timed out. stat = %04x, cntrl = %04x. Resetting master...\n",
		ioread32(I2C_REG_STS(alg_data)),
		ioread32(I2C_REG_CTL(alg_data)));

	clk_enable(alg_data->clk);

	/* Reset master and disable interrupts */
	ctl = ioread32(I2C_REG_CTL(alg_data));
	ctl &= ~(mcntrl_afie | mcntrl_naie | mcntrl_tdie | mcntrl_drmie);
	iowrite32(ctl, I2C_REG_CTL(alg_data));

	ctl |= mcntrl_reset;
	iowrite32(ctl, I2C_REG_CTL(alg_data));
	wait_reset(I2C_PNX_TIMEOUT, alg_data);
	alg_data->mif.ret = -EIO;
	complete(&alg_data->mif.complete);

	clk_disable(alg_data->clk);
}

static inline void bus_reset_if_active(struct i2c_pnx_algo_data *alg_data)
{
	u32 stat;

	if ((stat = ioread32(I2C_REG_STS(alg_data))) & mstatus_active) {
		dev_dbg(&alg_data->adapter.dev,
			"%s: Bus is still active after xfer. Sending Stop...\n",
			alg_data->adapter.name);
		i2c_pnx_stop(alg_data);
		if ((stat = ioread32(I2C_REG_STS(alg_data))) & mstatus_active) {
			dev_err(&alg_data->adapter.dev,
				"%s: Bus is still active after xfer, even after Stop. Reset it...\n",
				alg_data->adapter.name);
			i2c_pnx_recover(alg_data);
			iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_reset,
				  I2C_REG_CTL(alg_data));
			wait_reset(I2C_PNX_TIMEOUT, alg_data);
		}
	} else if (!(stat & mstatus_rfe) || !(stat & mstatus_tfe)) {
		/* If there is data in the fifo's after transfer,
		 * flush fifo's by reset.
		 */
		dev_dbg(&alg_data->adapter.dev,
			"%s: FIFOs not empty after xfer. Reset controller...\n",
			alg_data->adapter.name);
		iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_reset,
			  I2C_REG_CTL(alg_data));
		wait_reset(I2C_PNX_TIMEOUT, alg_data);
	} else if (stat & mstatus_nai) {
		dev_dbg(&alg_data->adapter.dev,
			"%s: Slave did not acknowledge, generating a STOP.\n",
			alg_data->adapter.name);
		i2c_pnx_stop(alg_data);
		iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_reset,
			  I2C_REG_CTL(alg_data));
		wait_reset(I2C_PNX_TIMEOUT, alg_data);
	}
}

/**
 * i2c_pnx_xfer - generic transfer entry point
 * @adap:		pointer to I2C adapter structure
 * @msgs:		array of messages
 * @num:		number of messages
 *
 * Initiates the transfer
 */
static int
i2c_pnx_xfer(struct i2c_adapter *adap, struct i2c_msg *msgs, int num)
{
	struct i2c_msg *pmsg;
	int rc = 0, completed = 0, i;
	struct i2c_pnx_algo_data *alg_data = adap->algo_data;
	u32 stat = ioread32(I2C_REG_STS(alg_data));

	dev_dbg(&alg_data->adapter.dev,
		"%s(): entering: %d messages, stat = %04x.\n",
		__func__, num, ioread32(I2C_REG_STS(alg_data)));

	clk_enable(alg_data->clk);

	bus_reset_if_active(alg_data);

	/* Enable master interrupt */
	iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_afie |
			mcntrl_naie,
		  I2C_REG_CTL(alg_data));

	/* Process transactions in a loop. */
	for (i = 0; rc >= 0 && i < num; i++) {
		u8 addr;

		pmsg = &msgs[i];
		addr = pmsg->addr;

		if (pmsg->flags & I2C_M_TEN) {
			dev_err(&alg_data->adapter.dev,
				"%s: 10 bits addr not supported!\n",
				alg_data->adapter.name);
			rc = -EINVAL;
			break;
		}

		alg_data->mif.buf = pmsg->buf;
		alg_data->mif.len = pmsg->len;
		alg_data->mif.dummy = pmsg->len;
		alg_data->mif.mode = (pmsg->flags & I2C_M_RD) ?
			I2C_SMBUS_READ : I2C_SMBUS_WRITE;
		alg_data->mif.ret = 0;
		alg_data->last = (i == num - 1);

		dev_dbg(&alg_data->adapter.dev, "%s(): mode %d, %d bytes\n",
			__func__, alg_data->mif.mode, alg_data->mif.len);

		i2c_pnx_arm_timer(alg_data);

		/* initialize the completion var */
		init_completion(&alg_data->mif.complete);

		/* Disable master interrupt */
		iowrite32(ioread32(I2C_REG_CTL(alg_data)) & ~(mcntrl_drmie |
				mcntrl_tdie),
			  I2C_REG_CTL(alg_data));

		/* Put start-code and slave-address on the bus. */
		rc = i2c_pnx_start(addr, alg_data);
		if (rc < 0)
			break;

		if (alg_data->last || (alg_data->mif.len > 0)) {
			/* Enable master interrupt */
			iowrite32(ioread32(I2C_REG_CTL(alg_data)) | mcntrl_drmie |
					mcntrl_tdie,
			I2C_REG_CTL(alg_data));

			/* Wait for completion */
			wait_for_completion(&alg_data->mif.complete);
		}

		if (!(rc = alg_data->mif.ret))
			completed++;
		dev_dbg(&alg_data->adapter.dev,
			"%s(): Complete, return code = %d.\n",
			__func__, rc);

		/* Clear TDI and AFI bits in case they are set. */
		if ((stat = ioread32(I2C_REG_STS(alg_data))) & mstatus_tdi) {
			dev_err(&alg_data->adapter.dev,
				"%s: TDI still set... clearing now.\n",
				alg_data->adapter.name);
			iowrite32(stat, I2C_REG_STS(alg_data));
		}
		if ((stat = ioread32(I2C_REG_STS(alg_data))) & mstatus_afi) {
			dev_err(&alg_data->adapter.dev,
				"%s: AFI still set... clearing now.\n",
				alg_data->adapter.name);
			iowrite32(stat, I2C_REG_STS(alg_data));
		}
	}

	bus_reset_if_active(alg_data);

	/* Cleanup to be sure... */
	alg_data->mif.buf = NULL;
	alg_data->mif.len = 0;

	dev_dbg(&alg_data->adapter.dev, "%s(): exiting, stat = %x\n",
		__func__, ioread32(I2C_REG_STS(alg_data)));

	clk_disable(alg_data->clk);

	if (completed != num)
		return ((rc < 0) ? rc : -EREMOTEIO);

	return num;
}

static u32 i2c_pnx_func(struct i2c_adapter *adapter)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL;
}

static struct i2c_algorithm pnx_algorithm = {
	.master_xfer = i2c_pnx_xfer,
	.functionality = i2c_pnx_func,
};

static int __devinit i2c_pnx_probe(struct platform_device *pdev)
{
	unsigned long tmp;
	int ret = 0;
	struct i2c_pnx_algo_data *alg_data;
	unsigned long freq;
	struct i2c_pnx_data *i2c_pnx = pdev->dev.platform_data;

	if (!i2c_pnx || !i2c_pnx->name) {
		dev_err(&pdev->dev, "%s: no platform data supplied\n",
		       __func__);
		ret = -EINVAL;
		goto out;
	}

	alg_data = kzalloc(sizeof(*alg_data), GFP_KERNEL);
	if (!alg_data) {
		ret = -ENOMEM;
		goto err_kzalloc;
	}

	platform_set_drvdata(pdev, alg_data);

	strlcpy(alg_data->adapter.name, i2c_pnx->name,
		sizeof(alg_data->adapter.name));
	alg_data->adapter.dev.parent = &pdev->dev;
	alg_data->adapter.algo = &pnx_algorithm;
	alg_data->adapter.algo_data = alg_data;
	alg_data->adapter.nr = pdev->id;
	alg_data->i2c_pnx = i2c_pnx;

	alg_data->clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(alg_data->clk)) {
		ret = PTR_ERR(alg_data->clk);
		goto out_drvdata;
	}

	init_timer(&alg_data->mif.timer);
	alg_data->mif.timer.function = i2c_pnx_timeout;
	alg_data->mif.timer.data = (unsigned long)alg_data;

	/* Register I/O resource */
	if (!request_mem_region(i2c_pnx->base, I2C_PNX_REGION_SIZE,
				pdev->name)) {
		dev_err(&pdev->dev,
		       "I/O region 0x%08x for I2C already in use.\n",
		       i2c_pnx->base);
		ret = -ENODEV;
		goto out_clkget;
	}

	alg_data->ioaddr = ioremap(i2c_pnx->base, I2C_PNX_REGION_SIZE);
	if (!alg_data->ioaddr) {
		dev_err(&pdev->dev, "Couldn't ioremap I2C I/O region\n");
		ret = -ENOMEM;
		goto out_release;
	}

	ret = clk_enable(alg_data->clk);
	if (ret)
		goto out_unmap;

	freq = clk_get_rate(alg_data->clk);

	/*
	 * Clock Divisor High This value is the number of system clocks
	 * the serial clock (SCL) will be high.
	 * For example, if the system clock period is 50 ns and the maximum
	 * desired serial period is 10000 ns (100 kHz), then CLKHI would be
	 * set to 0.5*(f_sys/f_i2c)-2=0.5*(20e6/100e3)-2=98. The actual value
	 * programmed into CLKHI will vary from this slightly due to
	 * variations in the output pad's rise and fall times as well as
	 * the deglitching filter length.
	 */

	tmp = ((freq / 1000) / I2C_PNX_SPEED_KHZ) / 2 - 2;
	if (tmp > 0x3FF)
		tmp = 0x3FF;
	iowrite32(tmp, I2C_REG_CKH(alg_data));
	iowrite32(tmp, I2C_REG_CKL(alg_data));

	tmp = (freq / 1000) / 1534;
	iowrite32(tmp, I2C_REG_HDA(alg_data));

	iowrite32(mcntrl_reset, I2C_REG_CTL(alg_data));
	if (wait_reset(I2C_PNX_TIMEOUT, alg_data)) {
		ret = -ENODEV;
		goto out_clock;
	}
	init_completion(&alg_data->mif.complete);

	ret = request_irq(i2c_pnx->irq, i2c_pnx_interrupt,
			0, pdev->name, alg_data);
	if (ret)
		goto out_clock;

	/* Register this adapter with the I2C subsystem */
	ret = i2c_add_numbered_adapter(&alg_data->adapter);
	if (ret < 0) {
		dev_err(&pdev->dev, "I2C: Failed to add bus\n");
		goto out_irq;
	}

	dev_dbg(&pdev->dev, "%s: Master at %#8x, irq %d.\n",
	       alg_data->adapter.name, i2c_pnx->base, i2c_pnx->irq);

	clk_disable(alg_data->clk);

	return 0;

out_irq:
	free_irq(i2c_pnx->irq, alg_data);
out_clock:
	clk_disable(alg_data->clk);
out_unmap:
	iounmap(alg_data->ioaddr);
out_release:
	release_mem_region(i2c_pnx->base, I2C_PNX_REGION_SIZE);
out_clkget:
	clk_put(alg_data->clk);
out_drvdata:
	kfree(alg_data);
err_kzalloc:
	platform_set_drvdata(pdev, NULL);
out:
	return ret;
}

static int __devexit i2c_pnx_remove(struct platform_device *pdev)
{
	struct i2c_pnx_algo_data *alg_data = platform_get_drvdata(pdev);
	struct i2c_pnx_data *i2c_pnx = alg_data->i2c_pnx;

	free_irq(i2c_pnx->irq, alg_data);
	i2c_del_adapter(&alg_data->adapter);
	iounmap(alg_data->ioaddr);
	release_mem_region(i2c_pnx->base, I2C_PNX_REGION_SIZE);
	clk_put(alg_data->clk);
	kfree(alg_data);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static struct platform_driver i2c_pnx_driver = {
	.driver = {
		.name = "pnx-i2c",
		.owner = THIS_MODULE,
	},
	.probe = i2c_pnx_probe,
	.remove = __devexit_p(i2c_pnx_remove),
};

static int __init i2c_adap_pnx_init(void)
{
	return platform_driver_register(&i2c_pnx_driver);
}

static void __exit i2c_adap_pnx_exit(void)
{
	platform_driver_unregister(&i2c_pnx_driver);
}

MODULE_AUTHOR("Vitaly Wool, Dennis Kovalev <source@mvista.com>");
MODULE_DESCRIPTION("I2C driver for Philips IP3204-based I2C busses");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:pnx-i2c");

/* We need to make sure I2C is initialized before USB */
subsys_initcall(i2c_adap_pnx_init);
module_exit(i2c_adap_pnx_exit);
