/*
 *  linux/drivers/staging/dspg/ectp0700_ts/ectp0700_ts.c
 *
 *  Copyright (C) 2012 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/wakelock.h>
#include <linux/earlysuspend.h>
#include <linux/suspend.h>
#include <linux/irq.h>
#include <linux/gpio.h>
#include <linux/ectp0700_ts.h>

#define ECTP0700_TS_I2C_NAME "ectp0700"

#define ABS_X_MIN 0
#define ABS_X_MAX 800
#define ABS_Y_MIN 0
#define ABS_Y_MAX 480

#define I2C_STARTTCH_READ 0xF9

#define MAX_POINTS 5

#define PROTOCOL_LEN 26

struct ectp0700_ts_touch_point {
	unsigned short w_tp_x;
	unsigned short w_tp_y;
};

struct ectp0700_ts_data {
	struct ectp0700_ts_platform_data *pdata;
	struct i2c_client *client;
	struct input_dev *input_dev;
	struct workqueue_struct *wq;
	struct work_struct work;
	struct early_suspend early_suspend;
	int irq;
};

static void
ectp0700_ts_work_func(struct work_struct *work)
{
	int i, x, y, s_w, s_h, point_num;
	struct ectp0700_ts_data *ts = container_of(work, struct ectp0700_ts_data, work);
	unsigned char buf[PROTOCOL_LEN] = { 0 };
	unsigned short low_byte, high_byte;
	unsigned char ecc = 0;
	struct ectp0700_ts_touch_point point[MAX_POINTS];

	if (i2c_smbus_read_i2c_block_data(ts->client, I2C_STARTTCH_READ,
	                                  PROTOCOL_LEN, buf) != PROTOCOL_LEN)
		return;

	/* check packet head: 0xAAAA */
	if (buf[1] != 0xaa || buf[0] != 0xaa)
		return;

	/* check data length */
	if ((buf[2] & 0x3f) != PROTOCOL_LEN)
		return;

	/* check number of points */
	point_num = buf[3] & 0x0f;
	if (point_num > MAX_POINTS)
		return;

	for (i = 0; i < point_num; i++) {
		high_byte = buf[5+4*i];
		low_byte = buf[5+4*i+1];
		point[i].w_tp_x = ((high_byte << 8) | low_byte) & 0x0fff;

		high_byte = buf[5+4*i+2];
		low_byte = buf[5+4*i+3];
		point[i].w_tp_y = ((high_byte << 8) | low_byte) & 0x0fff;
	}

	for (i = 0; i < PROTOCOL_LEN-1; i++)
		ecc ^= buf[i];

	if (ecc != buf[PROTOCOL_LEN-1])
		return;

	if (point_num <= 0) {
		input_report_abs(ts->input_dev, ABS_MT_TOUCH_MAJOR, 0);
		input_report_abs(ts->input_dev, ABS_MT_WIDTH_MAJOR, 5);
		input_mt_sync(ts->input_dev);
		input_sync(ts->input_dev);
	}

	for (i = 0; i < point_num; i++) {
		input_report_abs(ts->input_dev, ABS_MT_TOUCH_MAJOR, 200);
		input_report_abs(ts->input_dev, ABS_MT_WIDTH_MAJOR, 5);

		s_w = ABS_X_MAX - ABS_X_MIN;
		s_h = ABS_Y_MAX - ABS_Y_MIN;
		if (ts->pdata) {
			s_w = ts->pdata->screen_width;
			s_h = ts->pdata->screen_height;
		}
		x = point[i].w_tp_x * s_w / ABS_X_MAX;
		y = point[i].w_tp_y * s_h / ABS_Y_MAX;

		input_report_abs(ts->input_dev, ABS_MT_POSITION_X, x);
		input_report_abs(ts->input_dev, ABS_MT_POSITION_Y, y);
		input_mt_sync(ts->input_dev);
	}

	input_sync(ts->input_dev);
}

static irqreturn_t
ectp0700_ts_irq_handler(int irq, void *dev_id)
{
	struct ectp0700_ts_data *ts = (struct ectp0700_ts_data *)dev_id;

	queue_work(ts->wq, &ts->work);

	return IRQ_HANDLED;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void
ectp0700_ts_early_suspend(struct early_suspend *es)
{
	struct ectp0700_ts_data *ts = container_of(es,
	                              struct ectp0700_ts_data, early_suspend);

	if (ts->pdata && gpio_is_valid(ts->pdata->gpio_shutdown))
		gpio_set_value(ts->pdata->gpio_shutdown, 0);
}

static void
ectp0700_ts_late_resume(struct early_suspend *es)
{
	struct ectp0700_ts_data *ts = container_of(es,
	                              struct ectp0700_ts_data, early_suspend);

	if (ts->pdata && gpio_is_valid(ts->pdata->gpio_shutdown)) {
		gpio_set_value(ts->pdata->gpio_shutdown, 1);
		msleep(5);
		gpio_set_value(ts->pdata->gpio_shutdown, 0);
		msleep(5);
		gpio_set_value(ts->pdata->gpio_shutdown, 1);
	}
}
#endif

static int
ectp0700_ts_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct ectp0700_ts_data *ts;
	int x_max, y_max;

	if (!i2c_check_functionality(client->adapter,
	                             I2C_FUNC_SMBUS_READ_I2C_BLOCK)) {
		dev_err(&client->dev, "insufficient i2c functionality\n");
		return -ENODEV;
	}

	ts = kmalloc(sizeof(*ts), GFP_KERNEL);
	if (!ts) {
		dev_err(&client->dev, "memory allocation failed\n");
		return -ENOMEM;
	}

	ts->pdata = client->dev.platform_data;
	ts->wq = create_singlethread_workqueue("tp0700_wq");
	if (!ts->wq) {
		ret = -ENOMEM;
		goto create_work_queue_failed;
	}
	INIT_WORK(&ts->work, ectp0700_ts_work_func);

	ts->client = client;
	i2c_set_clientdata(client, ts);

	ts->input_dev = input_allocate_device();
	if (ts->input_dev == NULL) {
		ret = -ENOMEM;
		dev_err(&client->dev, "failed to allocate input device\n");
		goto input_dev_alloc_failed;
	}
	ts->input_dev->name = "ectp0700_ts";
	set_bit(EV_SYN, ts->input_dev->evbit);
	set_bit(EV_KEY, ts->input_dev->evbit);
	set_bit(BTN_TOUCH, ts->input_dev->keybit);
	set_bit(BTN_2, ts->input_dev->keybit);
	set_bit(EV_ABS, ts->input_dev->evbit);

	x_max = ABS_X_MAX;
	y_max = ABS_Y_MAX;
	if (ts->pdata) {
		x_max = ts->pdata->screen_width;
		y_max = ts->pdata->screen_height;
	}
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_X, ABS_X_MIN, x_max, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_Y, ABS_Y_MIN, y_max, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_TOUCH_MAJOR, 0, 255, 2, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_WIDTH_MAJOR, 0, 10, 2, 0);

	ret = input_register_device(ts->input_dev);
	if (ret) {
		dev_err(&client->dev, "unable to register input device\n");
		goto input_register_device_failed;
	}

	if (ts->pdata && gpio_is_valid(ts->pdata->gpio_shutdown)) {
		ret = gpio_request(ts->pdata->gpio_shutdown, ECTP0700_TS_I2C_NAME);
		if (ret) {
			dev_err(&client->dev, "unable to request shutdown GPIO\n");
			goto gpio_failed;
		}
		gpio_direction_output(ts->pdata->gpio_shutdown, 1);
		msleep(5);
		gpio_set_value(ts->pdata->gpio_shutdown, 0);
		msleep(5);
		gpio_set_value(ts->pdata->gpio_shutdown, 1);
	}

	ts->irq = client->irq;
	ret = request_irq(ts->irq, ectp0700_ts_irq_handler,
	                  IRQ_TYPE_EDGE_FALLING, ECTP0700_TS_I2C_NAME, ts);
	if (ret) {
		dev_err(&client->dev, "failed to register irq\n");
		ret = -EIO;
		goto request_irq_failed;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
	ts->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
	ts->early_suspend.suspend = ectp0700_ts_early_suspend;
	ts->early_suspend.resume = ectp0700_ts_late_resume;
	register_early_suspend(&ts->early_suspend);
#endif

	return 0;

request_irq_failed:
	if (ts->pdata && gpio_is_valid(ts->pdata->gpio_shutdown)) {
		gpio_set_value(ts->pdata->gpio_shutdown, 0);
		gpio_free(ts->pdata->gpio_shutdown);
	}
gpio_failed:
	input_unregister_device(ts->input_dev);
input_register_device_failed:
	input_free_device(ts->input_dev);
input_dev_alloc_failed:
	destroy_workqueue(ts->wq);
create_work_queue_failed:
	kfree(ts);

	return ret;
}

static int
ectp0700_ts_remove(struct i2c_client *client)
{
	struct ectp0700_ts_data *ts = i2c_get_clientdata(client);

	unregister_early_suspend(&ts->early_suspend);
	free_irq(ts->irq, ts);
	input_unregister_device(ts->input_dev);
	if (ts->pdata && gpio_is_valid(ts->pdata->gpio_shutdown)) {
		gpio_set_value(ts->pdata->gpio_shutdown, 0);
		gpio_free(ts->pdata->gpio_shutdown);
	}

	input_free_device(ts->input_dev);
	destroy_workqueue(ts->wq);
	kfree(ts);
	i2c_set_clientdata(client, NULL);

	return 0;
}

static const struct i2c_device_id ectp0700_ts_i2c_id[] = {
	{ ECTP0700_TS_I2C_NAME, 0 },
	{ }
};

static struct i2c_driver ectp0700_ts_driver = {
	.probe		= ectp0700_ts_probe,
	.remove		= ectp0700_ts_remove,
	.id_table	= ectp0700_ts_i2c_id,
	.driver		= {
		.name	= ECTP0700_TS_I2C_NAME,
	},
};

static int __devinit
ectp0700_ts_init(void)
{
	return i2c_add_driver(&ectp0700_ts_driver);
}

static void __exit
ectp0700_ts_exit(void)
{
	i2c_del_driver(&ectp0700_ts_driver);
}

module_init(ectp0700_ts_init);
module_exit(ectp0700_ts_exit);

MODULE_DESCRIPTION("TP0700 Touchscreen Driver");
MODULE_LICENSE("GPL");

